package com.calpano.common.client.crashreport;

import org.xydra.annotations.CanBeNull;
import org.xydra.env.IEnvironment;
import org.xydra.log.api.Logger;
import org.xydra.log.api.LoggerFactory;

import com.calpano.common.client.ClientApp;
import com.calpano.common.client.util.ClientUrl;
import com.google.gwt.http.client.Request;
import com.google.gwt.http.client.RequestBuilder;
import com.google.gwt.http.client.RequestCallback;
import com.google.gwt.http.client.RequestException;
import com.google.gwt.http.client.Response;
import com.google.gwt.user.client.Window.Location;

import de.xam.p13n.shared.time.TimeProvider;

/**
 * Utility to send errors via AJAX to the server.
 *
 * @author xamde
 */
public class ClientErrorHandlingUtils {

	private static final Logger log = LoggerFactory.getLogger(ClientErrorHandlingUtils.class);

	/**
	 * @param env
	 * @param msg
	 * @param jsUrl
	 * @param line
	 * @param consoleLogs
	 * @return true to let error handling continue
	 */
	public static boolean handleError(final IEnvironment env, final String msg, final String jsUrl,
			final String line, @CanBeNull final String consoleLogs) {
		final long timestamp = TimeProvider.getCurrentTimeInMillis();
		log.info("Sending error report ...");
		final ClientUrl cu = ClientUrl.fromCurrentLocation();
		cu.setPath("/clientError/" + timestamp);

		final ClientUrl q = ClientUrl.fromCurrentLocation();
		q.setQueryString("");
		q.setHash("");
		if (msg != null) {
			q.addQueryParamAndEncodeValue("msg", msg);
		}
		if (jsUrl != null) {
			q.addQueryParamAndEncodeValue("jsUrl", jsUrl);
		}
		if (line != null) {
			q.addQueryParamAndEncodeValue("line", line);
		}

		// add more data from client
		final String docUrl = Location.getHref();
		q.addQueryParamAndEncodeValue("docUrl", docUrl);
		q.addQueryParamAndEncodeValue("consoleLogs", consoleLogs);

		final RequestBuilder rb = new RequestBuilder(RequestBuilder.POST, cu.toString());
		rb.setHeader("Content-Type", "application/x-www-form-urlencoded");
		rb.setTimeoutMillis(60 * 1000);
		rb.setRequestData(q.getQueryString());
		rb.setCallback(new RequestCallback() {

			@Override
			public void onResponseReceived(final Request request, final Response response) {
				log.info("Client-side error logged on server");
			}

			@Override
			public void onError(final Request request, final Throwable exception) {
				log.warn("Failed to log client-side error on server. Retrying in another way...");
				handleErrorRobust(msg, jsUrl, line);
			}
		});
		try {
			rb.send();
		} catch (final RequestException e) {
			log.warn("Failed to log client-side error on server. Retrying in another way...", e);
			handleErrorRobust(msg, jsUrl, line);
		}

		log.info("Rebooting...");
		ClientApp.rebootOnError(env);
		return true;
	}

	/**
	 * @param env
	 * @param msg
	 * @param jsUrl
	 * @param line
	 * @return true to let error handling continue
	 */
	public boolean handleError(final IEnvironment env, final String msg, final String jsUrl,
			final String line) {
		// TODO remove this
		final String consoleLogs = ClientExceptionHandler.rememberLogListener.getLogs();
		return handleError(env, msg, jsUrl, line, consoleLogs);
	}

	public static native boolean handleErrorRobustNative(String shortMsg, String jsUrl, String line)
	/*-{
	    var timestamp = new Date().getTime();
	    var img=new Image();
	    img.src="/clientError/"+ timestamp + "?"+
	                 "shortMsg="+encodeURIComponent(shortMsg)+
	                 "&line="+encodeURIComponent(line)+
	                 "&jsUrl="+encodeURIComponent(jsUrl)+
	                 "&docUrl="+encodeURIComponent(document.URL);
	    return true;
	}-*/;

	/**
	 * @param msg
	 *            should be no longer than ca. 500 chars
	 * @param jsUrl
	 * @param line
	 * @return true to let error handling continue
	 */
	public static boolean handleErrorRobust(final String msg, final String jsUrl, final String line) {
		final String shortMsg = msg.substring(0, Math.min(msg.length(), 500)) + "...(truncated)";
		handleErrorRobustNative(shortMsg, jsUrl, line);
		return true;
	}

}
