package com.calpano.common.client.view.forms.impl;

import org.xydra.annotations.NativeCode;
import org.xydra.log.api.Logger;
import org.xydra.log.api.LoggerFactory;

import com.calpano.common.client.view.forms.activation.ActivationEvent;
import com.calpano.common.client.view.forms.activation.ActivationHandler;
import com.calpano.common.client.view.forms.activation.DeactivationEvent;
import com.calpano.common.client.view.forms.activation.DeactivationHandler;
import com.google.gwt.core.client.Scheduler;
import com.google.gwt.core.client.Scheduler.ScheduledCommand;
import com.google.gwt.dom.client.Element;
import com.google.gwt.dom.client.Style;
import com.google.gwt.dom.client.Style.Overflow;
import com.google.gwt.dom.client.Style.Position;
import com.google.gwt.dom.client.Style.Unit;
import com.google.gwt.user.client.Event;
import com.google.gwt.user.client.ui.RootPanel;
import com.google.gwt.user.client.ui.TextArea;

/**
 * A textarea that automatically expands its height to fit the text entered.
 * Uses a shadow textarea to calculate the height.
 */
@NativeCode
public class AutogrowTextArea extends Html5TextArea implements ActivationHandler,
		DeactivationHandler {

	private static final Logger log = LoggerFactory.getLogger(AutogrowTextArea.class);

	/**
	 * Copy the given CSS property from source to target
	 *
	 * @param cssPropertyName
	 * @param source
	 * @param target
	 */
	private static void matchStyles(final String cssPropertyName, final Style source, final Style target) {
		final String value = source.getProperty(cssPropertyName);
		if (value != null) {
			try {
				target.setProperty(cssPropertyName, value);
			} catch (final Exception e) {
				/*
				 * we might have a bogus value (e.g. width: -10px). we just let
				 * it fail quietly.
				 */
			}
		}
	}

	/** used to restore initial height when autogrow gets disabled */
	private String initialHeight;

	/** Not shown to user, used to calculate height. */
	private TextArea shadow;

	private int additionalEventsToSink;

	public AutogrowTextArea() {
		super();
		addActivationHandler(this);
		addDeactivationHandler(this);
	}

	public void adjustHeightDeferred() {
		Scheduler.get().scheduleDeferred(new ScheduledCommand() {

			@Override
			public void execute() {
				AutogrowTextArea.this.adjustHeightNow();
			}
		});

	}

	/** called only via deferred handler */
	private void adjustHeightNow() {
		/*
		 * scrollHeight := An element's scrollHeight is a measurement of the
		 * height of an element's content including content not visible on the
		 * screen due to overflow.
		 */

		/** prepare shadow */
		this.shadow.setWidth(getElement().getScrollWidth() + 12 + "px");
		this.shadow.setHeight("auto");
		/** set text in shadow area */
		this.shadow.setText(getValue());
		this.shadow.setHeight(this.shadow.getElement().getScrollHeight() + 20 + "px");
		/** propagate height determined by browser back to us */
		setHeight(this.shadow.getElement().getScrollHeight() + 18 + "px");
	}

	/**
	 * Autogrow is en-/disabled to not pollute the DOM with too many shadow
	 * textareas
	 */
	@Override
	public void onActivation(final ActivationEvent event) {
		log.trace("activating autogrow");
		final Element element = getElement();
		final Style thisStyle = element.getStyle();

		this.shadow = new TextArea();
		RootPanel.get().add(this.shadow);

		// adapt this style
		thisStyle.setOverflow(Overflow.HIDDEN);
		thisStyle.setProperty("resize", "none");
		this.initialHeight = thisStyle.getProperty("height");

		// adapt shadow style
		final Style shadowStyle = this.shadow.getElement().getStyle();
		// position shadow textarea outside of visible screen area
		shadowStyle.setPosition(Position.ABSOLUTE);
		shadowStyle.setTop(-9999, Unit.PX);
		shadowStyle.setLeft(-9999, Unit.PX);
		shadowStyle.setProperty("bottom", "auto");
		shadowStyle.setProperty("right", "auto");
		matchStyles("display", thisStyle, shadowStyle);
		matchStyles("fontFamily", thisStyle, shadowStyle);
		matchStyles("fontSize", thisStyle, shadowStyle);
		matchStyles("fontStyle", thisStyle, shadowStyle);
		matchStyles("fontWeight", thisStyle, shadowStyle);
		matchStyles("letterSpacing", thisStyle, shadowStyle);
		matchStyles("lineHeight", thisStyle, shadowStyle);
		matchStyles("overflow", thisStyle, shadowStyle);
		matchStyles("paddingBottom", thisStyle, shadowStyle);
		matchStyles("paddingLeft", thisStyle, shadowStyle);
		matchStyles("paddingRight", thisStyle, shadowStyle);
		matchStyles("paddingTop", thisStyle, shadowStyle);
		matchStyles("resize", thisStyle, shadowStyle);
		matchStyles("textIndent", thisStyle, shadowStyle);
		matchStyles("textTransform", thisStyle, shadowStyle);
		matchStyles("width", thisStyle, shadowStyle);
		matchStyles("wordSpacing", thisStyle, shadowStyle);

		// start listening
		/** whenever the textarea changes, adjust its height */
		sinkEvents(this.additionalEventsToSink);
		registerOnCut(getElement());
	}

	@Override
	public void onDeactivation(final DeactivationEvent event) {
		// stop listening
		deregisterOnCut(getElement());
		unsinkEvents(this.additionalEventsToSink);
		// clean up
		this.shadow.removeFromParent();
		resetHeight();
		this.shadow = null;
	}

	@Override
	public void onBrowserEvent(final Event event) {
		if (log.isTraceEnabled()) {
			log.trace("Autogrow event " + event.getType());
		}

		super.onBrowserEvent(event);

		// switch(DOM.eventGetType(event)) {
		// case Event.ONBLUR:
		// case Event.ONKEYUP:
		// case Event.ONKEYPRESS:
		// case Event.ONPASTE:
		// adjustHeightDeferred();
		// break;
		// }
	}

	@Override
	public void onLoad() {
		super.onLoad();

		/* compute the bitfield of events required additionally */

		// 0 1 1 e.g. 0b10000001011110000011
		final int sunkEvents = super.getSunkEvents();

		// 1 1 0 e.g. 0b10000001001100000000
		final int sinkEvents = Event.ONPASTE | Event.ONKEYUP | Event.ONKEYPRESS | Event.ONBLUR;

		/*
		 * goal: for each bit in sinkEvent, check if it was already 1 in
		 * sunkEvents. If so, remove it.
		 */
		// 0 0 1 e.g. 0b10000000010010000011 =
		int reqEvents = 0;
		for (int n = 0; n < 32; n++) {
			final int marker = 1 << n;
			if ((marker & sinkEvents) > 0 && (marker & ~sunkEvents) > 0) {
				reqEvents |= marker;
			}
		}
		this.additionalEventsToSink = reqEvents;
	}

	private native void registerOnCut(Element element)
	/*-{

	element.oncut = $entry(function() {
	this.@com.calpano.common.client.view.forms.impl.AutogrowTextArea::adjustHeightDeferred()();
	return false;
	});

	}-*/;

	private native void deregisterOnCut(Element element)
	/*-{

	element.oncut = null;

	}-*/;

	/**
	 * Resets the height of the textarea to the initial height. Used when text
	 * in textarea is programmatically cleared.
	 */
	public void resetHeight() {
		if (getValue().isEmpty()) {
			getElement().getStyle().setProperty("height", this.initialHeight);
		}

	}

}
