package com.calpano.common.client.view.forms.impl;

import org.xydra.annotations.CanBeNull;
import org.xydra.annotations.Feature;
import org.xydra.annotations.NeverNull;
import org.xydra.conf.annotations.RequireConf;
import org.xydra.env.Env;
import org.xydra.log.api.Logger;
import org.xydra.log.api.LoggerFactory;

import com.calpano.common.client.util.SuggestionDisplay;
import com.calpano.common.client.view.forms.AutofocusManager;
import com.calpano.common.client.view.forms.Html5DomUtil;
import com.calpano.common.client.view.forms.IBelongsToHtml5Form;
import com.calpano.common.client.view.forms.IHtml5TextInput;
import com.calpano.common.client.view.forms.ViewConstants;
import com.calpano.common.client.view.forms.activation.H5Activation;
import com.calpano.common.client.view.forms.placeholder.IPlaceholderSupport;
import com.calpano.common.client.view.forms.suggestion.HideSuggestionEvent;
import com.calpano.common.client.view.forms.suggestion.HideSuggestionHandler;
import com.calpano.common.client.view.forms.suggestion.ShowSuggestionEvent;
import com.calpano.common.client.view.forms.suggestion.ShowSuggestionHandler;
import com.calpano.common.client.view.forms.suggestion.impl.SuggestionManager;
import com.calpano.common.client.view.forms.utils.EventUtil;
import com.calpano.common.client.view.forms.validation.InvalidationHandler;
import com.calpano.common.client.view.forms.validation.ValidationHandler;
import com.calpano.common.client.view.forms.validation.impl.CalculateValidationHelper;
import com.calpano.common.client.view.forms.validation.impl.ShowValidationMessageHelper;
import com.calpano.common.shared.ConfParamsCalpanoCommonShared;
import com.calpano.common.shared.validation.IStringValidator;
import com.calpano.common.shared.validation.ValidationMessage;
import com.google.gwt.event.dom.client.KeyCodes;
import com.google.gwt.event.logical.shared.ValueChangeEvent;
import com.google.gwt.user.client.DOM;
import com.google.gwt.user.client.Event;
import com.google.gwt.user.client.ui.RootPanel;
import com.google.gwt.user.client.ui.SuggestOracle;
import com.google.gwt.user.client.ui.TextBoxBase;
import com.google.web.bindery.event.shared.HandlerRegistration;

/**
 * Shared code for validation-capable text widgets.
 *
 * Implementation: GWT widgets get extended and delegate all relevant calls to
 * an implementation of this interface. Extended GWT widgets implement
 * IHasHtml5TextBoxAndAreaPart, which in turn gives access to an
 * {@link IHtml5TextInput}.
 *
 * Delegates to a {@link PlaceholderHelper} for simulating 'placeholder'.
 *
 * Delegates to {@link CalculateValidationHelper} to simulate 'required' and
 * 'type=email'.
 *
 * Adds/removes the CSS class {@link ViewConstants#CLASS_INVALID} to the given
 * widget.
 *
 * Edits attributes 'type', 'required'.
 *
 * Implementation note: All focus-setting should be done by the container form,
 * otherwise the form elements battle to get the focus.
 *
 * @param <T>
 *            T base widget type
 */
class Html5TextBoxAndAreaPart<T extends TextBoxBase & IPlaceholderSupport> extends H5Activation<T>
		implements IHtml5TextInput, ShowSuggestionHandler, HideSuggestionHandler {

	private static final Logger log = LoggerFactory.getLogger(Html5TextBoxAndAreaPart.class);

	/** implements {@link IBelongsToHtml5Form} */
	private Html5FormPanel form;

	private final ShowValidationMessageHelper showValidationMessageHelper;

	private final boolean submitOnEnter;

	@Feature("validation")
	private final CalculateValidationHelper validationHelper;

	@CanBeNull
	@Feature("suggestions")
	private SuggestionDisplay suggestionDisplay;

	@Feature("suggestions")
	private SuggestionManager suggestionManager;

	@Feature("suggestions")
	private boolean isShowingSuggestions;

	// TODO set debug flag in constructor
	@RequireConf(ConfParamsCalpanoCommonShared.clientDebugMode)
	public Html5TextBoxAndAreaPart(final T gwtTextBoxBase, final boolean submitOnEnter,
			final boolean simulatePlaceholder, final boolean simulateValidation) {
		super(gwtTextBoxBase, simulatePlaceholder);
		this.submitOnEnter = submitOnEnter;

		this.validationHelper = new CalculateValidationHelper(this, simulateValidation);
		this.showValidationMessageHelper = new ShowValidationMessageHelper(this);

		int eventBits = 0;
		// submit by enter
		eventBits |= Event.ONKEYUP;
		// activation, placeholder
		eventBits |= super.getEventsToSink();
		// validation warnings
		eventBits |= this.showValidationMessageHelper.getEventsToSink();
		// debug mode support
		final boolean debugMode = Env.get().conf()
				.getBoolean(ConfParamsCalpanoCommonShared.clientDebugMode);
		if (debugMode) {
			eventBits |= EventUtil.EVENT_USED;
		}

		getTextBoxBase().sinkEvents(eventBits);
	}

	@Override
	@Feature("validation")
	public void addValidator(final Integer order, @NeverNull final IStringValidator validator) {
		this.validationHelper.addValidator(order, validator);
	}

	/* implements {@link IBelongsToHtml5Form} */
	@Override
	public Html5FormPanel getForm() {
		return this.form;
	}

	@Override
	@Feature("validation")
	public String getType() {
		return this.validationHelper.getType();
	}

	@Feature("validation")
	@Override
	public ValidationMessage computeValidation() {
		return this.validationHelper.computeValidation();
	}

	/**
	 * @return true attribute 'required' is set
	 */
	@Override
	public boolean isRequired() {
		return this.validationHelper.isRequired();
	}

	@Override
	@RequireConf(ConfParamsCalpanoCommonShared.clientDebugMode)
	public void onBrowserEvent(final Event event) {
		/* support debug mode */
		// debug mode support
		final boolean debugMode = Env.get().conf()
				.getBoolean(ConfParamsCalpanoCommonShared.clientDebugMode);
		if (debugMode) {
			log.info("'''" + "INPUT" + ": " + EventUtil.dumpEvent(event));
		}

		/*
		 * also calculate placeholder stuff, which in turn fires validation
		 * events
		 */
		super.onBrowserEvent(event);

		if (isShowingSuggestions()) {
			this.suggestionManager.onBrowserEvent(event);
		} else {
			/* decide if the event triggers a submit */
			switch (DOM.eventGetType(event)) {
			case Event.ONKEYDOWN:
				switch (event.getKeyCode()) {
				case KeyCodes.KEY_ENTER: {
					if (this.submitOnEnter && !this.isShowingSuggestions()) {
						this.getForm().tryToSubmit(this);
						event.stopPropagation();
						event.preventDefault();
					}
				}
					break;
				case KeyCodes.KEY_DOWN:
					if (this.suggestionManager != null) {
						this.suggestionManager.showSuggestions("");
					}
					break;
				}
				break;
			}

			/* then also hand the event to decide what to show or hide */
			this.showValidationMessageHelper.onBrowserEvent(event);
		}
	}


	@Override
	public void onLoad() {
		super.onLoad();
		this.validationHelper.onLoad();

		// FIXME do not register yet
		// this.validationHelper.addValidityChangeHandler(this.showValidationMessageHelper);
	}

	@Override
	public void reset() {
		super.reset();
		getTextBoxBase().setText("");
		this.showValidationMessageHelper.reset();
		if (this.suggestionManager != null) {
			this.suggestionManager.reset();
		}
		this.validationHelper.reset();
	}

	/**
	 * HTML5 feature. See http://www.w3.org/wiki/HTML5_form_additions#autofocus
	 *
	 * There should only be one element with 'autofocus' per <em>page</em>.
	 *
	 * @param autofocus
	 */
	@Feature("autofocus")
	public void setAutofocus(final boolean autofocus) {
		AutofocusManager.setAutofocusWidget(getTextBoxBase(), autofocus);
	}

	/**
	 * HTML5 feature, see http://www.w3.org/wiki/HTML5_form_additions#autofocus
	 *
	 * There should only be one element with 'autofocus' per <em>page</em>.
	 *
	 * @param attributeValue
	 */
	@Feature("autofocus")
	public void setAutofocus(final String attributeValue) {
		if (attributeValue.equals("autofocus") || attributeValue.equals("")) {
			setAutofocus(true);
		}
	}

	/* implements {@link IBelongsToHtml5Form} */
	@Override
	public void setForm(final Html5FormPanel html5FormPanel) {
		this.form = html5FormPanel;
	}

	@Override
	@Feature("validation")
	public void setRequired(final boolean required) {
		this.validationHelper.setRequired(required);
	}

	/**
	 * HTML5 feature.
	 *
	 * @param attributeValue
	 */
	@Feature("validation")
	public void setRequired(final String attributeValue) {
		this.validationHelper.setRequired(attributeValue);
	}

	/**
	 * HTML5 feature.
	 *
	 * @param type
	 *            for type 'email' we have a very strict check
	 */
	@Feature("validation")
	public void setType(final String type) {
		this.validationHelper.setType(type);
	}

	/**
	 * Called just before this widget is removed in DOM from parent. UIBinder
	 * calls this also at startup when initialising widgets that are used in
	 * HTMLPanel.
	 */
	public void onBeforeRemoveFromParent() {
		this.showValidationMessageHelper.onBeforeRemoveFromParent();
	}

	@Override
	public void activateValidationWarnings() {
		this.showValidationMessageHelper.activateValidationWarnings();
	}

	/**
	 * See http://www.w3.org/TR/2011/WD-html5-20110525/forms.html#attr-form-
	 * autocomplete
	 *
	 * @param onOrOff
	 *            should be "on" or "off". "" means "on".
	 */
	@SuppressWarnings("null")
	public void setAutocomplete(final String onOrOff) {
		final boolean on = onOrOff == null || onOrOff.equals("") || onOrOff.equalsIgnoreCase("on");
		assert on || onOrOff.equalsIgnoreCase("off") : "autocomplete must be '','on' or 'off'";

		/* if on, it's the default anyway and we just don't touch the DOM */
		if (!on) {
			Html5DomUtil.setPropertyString(getTextBoxBase(), "autocomplete", "off");
		}
	}

	@Override
	public HandlerRegistration addValidationHandler(final ValidationHandler handler) {
		return this.validationHelper.addValidationHandler(handler);
	}

	@Override
	public HandlerRegistration addInvalidationHandler(final InvalidationHandler handler) {
		return this.validationHelper.addInvalidationHandler(handler);
	}

	@Override
	public boolean isShowingSuggestions() {
		return this.isShowingSuggestions;
	}

	@Override
	public void suggestValuesFrom(final SuggestOracle oracle) {
		// enable suggestions
		this.suggestionDisplay = new SuggestionDisplay();
		this.suggestionDisplay.setEnabled(true);
		RootPanel.get().add(this.suggestionDisplay);
		this.suggestionManager = new SuggestionManager(oracle, this.base, this.suggestionDisplay);
		this.suggestionManager.addShowSuggestionHandler(this);
		this.suggestionManager.addHideSuggestionHandler(this);
		getTextBoxBase().addValueChangeHandler(this);
	}

	@Override
	public void onShowSuggestion(final ShowSuggestionEvent event) {
		log.trace("is showing suggestions");
		this.isShowingSuggestions = true;
	}

	@Override
	public void onHideSuggestion(final HideSuggestionEvent event) {
		log.trace("is not showing suggestions");
		this.isShowingSuggestions = false;
	}

	@Override
	public void onValueChange(final ValueChangeEvent<String> event) {
		super.onValueChange(event);
		if (this.suggestionManager != null) {
			this.suggestionManager.onValueChange(event);
		}
	}

	public void onUnload() {
		// FIXME @Max suggestions & popups
		this.validationHelper.onUnload();
	}

	public void setSuggestionsEnabled(final boolean enabled) {
		assert this.suggestionManager != null;
		this.suggestionManager.setEnabled(enabled);
	}

}
