package com.calpano.common.client.view.forms.validation.impl;

import org.xydra.annotations.CanBeNull;
import org.xydra.log.api.Logger;
import org.xydra.log.api.LoggerFactory;

import com.calpano.common.client.view.forms.IHtml5TextInput;
import com.calpano.common.client.view.forms.StaticFormManager;
import com.calpano.common.client.view.forms.Tooltip;
import com.calpano.common.client.view.forms.Tooltip.TooltipPosition;
import com.calpano.common.client.view.forms.validation.InvalidationEvent;
import com.calpano.common.client.view.forms.validation.InvalidationHandler;
import com.calpano.common.client.view.forms.validation.ValidationEvent;
import com.calpano.common.client.view.forms.validation.ValidationHandler;
import com.calpano.common.client.view.resources.CommonResourceBundle;
import com.calpano.common.shared.validation.ValidationMessage;
import com.google.gwt.event.logical.shared.ValueChangeEvent;
import com.google.gwt.event.logical.shared.ValueChangeHandler;
import com.google.gwt.safehtml.shared.SafeHtmlUtils;
import com.google.gwt.user.client.Event;
import com.google.web.bindery.event.shared.HandlerRegistration;

/**
 * Listens to validation change events and shows or hides validation messages.
 */
public class ShowValidationMessageHelper implements ValidationHandler, InvalidationHandler {

	private static final Logger log = LoggerFactory.getLogger(ShowValidationMessageHelper.class);

	private final IHtml5TextInput source;

	@CanBeNull
	private Tooltip tooltip;

	private HandlerRegistration reg;

	/**
	 * @param source
	 */
	public ShowValidationMessageHelper(final IHtml5TextInput source) {
		this.source = source;
		CommonResourceBundle.INSTANCE.css().ensureInjected();
	}

	public int getEventsToSink() {
		return 0;
	}

	private boolean hasTooltip() {
		return this.tooltip != null;
	}

	private void hideValidationTooltip() {
		if (this.tooltip != null) {
			this.tooltip.hideTooltip();
			this.tooltip = null;
		}
	}

	private void showValidationTooltip() {
		final ValidationMessage vmsg = this.source.computeValidation();
		final String msg = vmsg.message;
		if (!hasTooltip()) {
			final String fullMsg = StaticFormManager.getStringTransformer().transform(msg);
			this.tooltip = new Tooltip().setHtml(SafeHtmlUtils.fromSafeConstant(fullMsg))
					.setPosition(TooltipPosition.BELOW_LEFT).attachTo(this.source.getTextBoxBase());

			this.tooltip.addStyleName(CommonResourceBundle.INSTANCE.css().validationTooltip());
			this.tooltip.showTooltipDelayed(30);
		}
	}

	private void hideBoth() {
		hideValidationTooltip();
		removeCssClassForWarning();
	}

	private void showBoth() {
		addCssClassForWarning();
		showValidationTooltip();
	}

	private void removeCssClassForWarning() {
		CommonResourceBundle.INSTANCE.css().ensureInjected();
		this.source.getTextBoxBase().removeStyleName(CommonResourceBundle.INSTANCE.css().invalid());
	}

	private void addCssClassForWarning() {
		CommonResourceBundle.INSTANCE.css().ensureInjected();
		this.source.getTextBoxBase().addStyleName(CommonResourceBundle.INSTANCE.css().invalid());
	}

	public void onBrowserEvent(final Event event) {
	}

	@Override
	public void onInvalid(final InvalidationEvent event) {
		log.debug("'" + this.source.getRawText() + "' = invalid");
		updateValidationWarnings();
	}

	@Override
	public void onValid(final ValidationEvent event) {
		log.debug("'" + this.source.getRawText() + "' = valid");
		updateValidationWarnings();
	}

	public void deactivateValidationWarnings() {
		if (this.reg != null) {
			this.reg.removeHandler();
			this.reg = null;
		}
		hideBoth();
	}

	public void activateValidationWarnings() {
		log.trace("activated validation warnings");
		// check now
		updateValidationWarnings();

		// listen from now on
		this.reg = this.source.getTextBoxBase().addValueChangeHandler(
				new ValueChangeHandler<String>() {

					@Override
					public void onValueChange(final ValueChangeEvent<String> event) {
						updateValidationWarnings();
					}
				});
	}

	private void updateValidationWarnings() {
		final ValidationMessage vmsg = this.source.computeValidation();
		if (vmsg.level.isValid()) {
			// we hide warnings ASAP
			hideBoth();
		} else {
			showBoth();
		}
	}

	public void reset() {
		deactivateValidationWarnings();
	}

	/**
	 * Remove tooltip from DOM
	 */
	public void onBeforeRemoveFromParent() {
		// which also removes it from DOM
		hideValidationTooltip();
	}

}
