package com.calpano.common.shared.mail;

import java.io.Serializable;

import org.xydra.annotations.CanBeNull;
import org.xydra.annotations.RequiresAppEngine;
import org.xydra.annotations.RunsInAppEngine;
import org.xydra.annotations.RunsInGWT;
import org.xydra.base.XId;
import org.xydra.log.api.Logger;
import org.xydra.log.api.LoggerFactory;

/**
 * Data container to model an email
 *
 * @author xamde
 *
 */
@RunsInGWT(true)
@RunsInAppEngine(true)
@RequiresAppEngine(false)
public class Mail implements Serializable {

	private static final Logger log = LoggerFactory.getLogger(Mail.class);

	public Mail() {
	}

	private static final long serialVersionUID = -2130057152854788754L;

	public String senderName;
	/** A single email address */
	public String senderMailAddress;

	public String recipientName;
	/** A single email address */
	public String recipientMailAddress;

	/** A single email address */
	public String bccMailAddress;

	public String subject;

	public String messageTextPlain;
	public String messageTextHtml;

	/** Can be null if not a known user */
	public XId recipientUserId;

	/**
	 * An internal runtime message, never included in the real email
	 */
	public String debugMsg;

	/**
	 * An internal runtime message, never included in the real email
	 */
	@CanBeNull
	public String generatedByTemplate;

	public boolean hasHybridBody() {
		return hasTextPlainBody() && hasTextHtmlBody();
	}

	public boolean hasTextHtmlBody() {
		return this.messageTextHtml != null;
	}

	public boolean hasTextPlainBody() {
		return this.messageTextPlain != null;
	}

	@Override
	public String toString() {
		return toPlainText();
	}

	public String toPlainText() {
		return

		"Debug: " + this.debugMsg + "\n"

		+ "Template: " + this.generatedByTemplate + " \n"

		+ "From: " + this.senderMailAddress + " (From-Name: " + this.senderName + ")\n"

		+ "To: " + this.recipientMailAddress + " (To-Name: " + this.recipientName + ")\n"

		+ "Bcc: " + this.bccMailAddress + "\n"

		+ "Subject: " + this.subject + "\n"

		+ "Message text/plain:\n ***************\n" + this.messageTextPlain + "\n***************\n"

		+ "Message text/html:\n ***************\n" + this.messageTextHtml + "\n***************\n";
	}

	/**
	 * @param raw
	 *            ..
	 * @return raw with all German Umlauts replaced: 'a with two points' becomes
	 *         'ae' etc.
	 */
	public static final String asciiEncode(final String raw) {
		String s = raw;
		s = s.replace("ä", "ae");
		s = s.replace("ö", "oe");
		s = s.replace("ü", "ue");
		s = s.replace("Ä", "Ae");
		s = s.replace("Ö", "Oe");
		s = s.replace("Ü", "Ue");
		s = s.replace("ß", "ss");
		s = s.replace("€", "EUR");
		return s;
	}

	/**
	 * @return true if mail has recipientMailAddress, subject, messageText*,
	 *         senderName, senderMailAdress
	 */
	public boolean hasMinimalData() {
		final StringBuffer problems = new StringBuffer();
		if (this.recipientMailAddress == null) {
			problems.append("address is null;");
		}
		/* recipientName allowed to be null */
		if (this.subject == null) {
			problems.append("subject is null;");
		}
		if (this.messageTextPlain == null && this.messageTextHtml == null) {
			problems.append("messageTextPlain AND messageTextHtml is null;");
		}
		if (this.senderName == null) {
			problems.append("senderName is null;");
		}
		if (this.senderMailAddress == null) {
			problems.append("senderMail is null;");
		}
		/** additional BCC allowed to be null */

		if (problems.length() == 0) {
			return true;
		} else {
			log.warn("Problems detected: " + problems.toString());
			return false;
		}
	}

	/**
	 * Helps to detect mal-formed email where Velocity variables could not get
	 * resolved.
	 *
	 * @return true if a dollar followed by a curly brace appears in any
	 *         relevant email field or text.
	 */
	public boolean containsDollarOpenCurlyBrace() {
		return containsDollarOpenCurlyBrace("bcc", this.bccMailAddress)
				|| containsDollarOpenCurlyBrace("html", this.messageTextHtml)
				|| containsDollarOpenCurlyBrace("plaintext", this.messageTextPlain)
				|| containsDollarOpenCurlyBrace("receiverMail", this.recipientMailAddress)
				|| containsDollarOpenCurlyBrace("receiverName", this.recipientName)
				|| containsDollarOpenCurlyBrace("senderMail", this.senderMailAddress)
				|| containsDollarOpenCurlyBrace("senderName", this.senderName)
				|| containsDollarOpenCurlyBrace("subject", this.subject);
	}

	/**
	 * @param part
	 *            where the check is performed
	 * @param s
	 *            can be null
	 * @return false if s is not null and contains curly braces
	 */
	public static boolean containsDollarOpenCurlyBrace(final String part, final String s) {
		if (s == null) {
			return false;
		}
		final boolean found = s.contains("${");
		if (found) {
			log.warn("Found '${' in mail part '" + part + "' = '" + s + "'");
		}
		return found;
	}

}
