package com.calpano.common.shared.util;

import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Set;
import java.util.SortedMap;
import java.util.SortedSet;
import java.util.TreeMap;
import java.util.TreeSet;

import org.xydra.restless.utils.SharedHtmlUtils;
import org.xydra.restless.utils.SharedHtmlUtils.Table;
import org.xydra.restless.utils.SharedHtmlUtils.Table.TableRow;

public class Stats {

	/**
	 * multiple views on the same base data
	 */
	public static class Views extends Metric {
		public Views(final String name) {
			super(name);
		}

		public void addView(final Metric m) {
			this.children.add(m);
		}

		@Override
		public String toHtml() {
			final List<String> htmls = new LinkedList<String>();
			for (final Metric m : this.children) {
				htmls.add(m.toHtml());
			}
			return SharedHtmlUtils.toOrderedList(htmls);
		}
	}

	public static abstract class Metric implements Comparable<Metric> {

		public Metric(final String name) {
			this.name = name;
		}

		String name;
		long value;

		public SortedSet<Metric> children = new TreeSet<Metric>();

		@Override
		public int compareTo(final Metric o) {
			return this.name.compareTo(o.name);
		}

		public String toHtml() {
			return this.name + ": " + this.value;
		}
	}

	/**
	 * Counts for each label the number of times it occurred in the dataset.
	 */
	public static class TagHistogramm extends Metric {

		private final SortedMap<String, Integer> map = new TreeMap<String, Integer>();

		public TagHistogramm(final String name) {
			super(name);
		}

		public void count(final String tag) {
			count(tag, 1);
		}

		public Set<Entry<String, Integer>> entrySet() {
			return this.map.entrySet();
		}

		public void count(final String tag, final int times) {
			Integer i = this.map.get(tag);
			if (i == null) {
				i = times;
			} else {
				i = i + times;
			}
			this.map.put(tag, i);
			super.value += times;
		}

		public void count(final String tag, final long times) {
			count(tag, (int) times);
		}

		public void addToDataMap(final Map<String, String> map) {
			for (final Entry<String, Integer> entry : entrySet()) {
				map.put(entry.getKey(), entry.getValue() + "");
			}
		}

		@Override
		public String toHtml() {
			final StringBuilder b = new StringBuilder();
			b.append(this.name + " = " + this.value);
			final Table table = SharedHtmlUtils.table();

			TableRow row = table.row();
			row.th("Tag");
			for (final Entry<String, Integer> entry : entrySet()) {
				final String raw = entry.getKey();
				final String safe = SharedHtmlUtils.sanitize(raw);
				row.th(safe);
			}

			row = table.row();
			row.th("Count");
			for (final Entry<String, Integer> entry : entrySet()) {
				row.td(entry.getValue() + "");
			}

			b.append(table.toStringBuilder());
			return b.toString();
		}

	}

	public static class EnumHistogramm<T extends Enum<?>> extends Metric {

		TagHistogramm h;

		public EnumHistogramm(final String name) {
			super(name);
			this.h = new TagHistogramm(name);
		}

		public void count(final T enumValue) {
			this.h.count(enumValue.name());
		}

		public void addToDataMap(final Map<String, String> map) {
			this.h.addToDataMap(map);
		}

		@Override
		public String toHtml() {
			return this.h.toHtml();
		}

	}

	/**
	 * @param n
	 *            how many digits
	 * @param number
	 *            to be formatted
	 * @return "007"
	 */
	public static String digits(final int n, final long number) {
		String s = "" + number;
		while (s.length() < n) {
			s = " " + s;
		}
		return s;
	}

	public static String digits(final int n, final int number) {
		return digits(n, (long) number);
	}

}
