package com.calpano.common.shared.validation;

import org.xydra.log.api.Logger;
import org.xydra.log.api.LoggerFactory;

import com.calpano.common.shared.mail.EmailUtils;

/**
 * Validate required or not; Types: None or Email.
 */
public class TextValidator implements IStringValidator {

	private static final Logger log = LoggerFactory.getLogger(TextValidator.class);

	private final boolean required;

	private final Type type;

	private final String invalidBadEmail;

	private final String invalidRequiredButEmpty;

	/** Supported built-in validation types */
	public static enum Type {
		None, Email
	}

	public TextValidator(final boolean required, final String typeStr, final String invalidRequiredButEmpty,
			final String invalidBadEmail) {
		this.required = required;
		this.type = TextValidator.parse(typeStr);
		this.invalidRequiredButEmpty = invalidRequiredButEmpty;
		this.invalidBadEmail = invalidBadEmail;
	}

	/**
	 * @param typeStr
	 *            casing is ignored
	 * @return one of the types in {@link Type} or {@link Type#None}
	 */
	public static Type parse(final String typeStr) {
		for (final Type t : Type.values()) {
			if (typeStr.toLowerCase().equalsIgnoreCase(t.name())) {
				return t;
			}
		}
		return Type.None;
	}

	@Override
	public ValidationMessage computeValidation(final String text) {
		log.trace("TextValidator: '" + text + "' = validEmail " + EmailUtils.isValidEmail(text));

		// check 'required'
		if (this.required) {
			if (text == null || text.equals("")) {
				return ValidationMessage.error(this.invalidRequiredButEmpty, false);
			}
		}

		switch (this.type) {
		case Email: {
			if (text.equals("") || EmailUtils.isValidEmail(text)) {
				// fine
			} else {
				return ValidationMessage.error(this.invalidBadEmail, true);
			}
		}
			break;
		case None:
			// fine
		}

		return ValidationMessage.ALL_OK;
	}

}
