package com.calpano.common.shared.xydrautils;

import java.util.Collection;
import java.util.HashSet;
import java.util.Iterator;

import org.xydra.base.XId;
import org.xydra.base.rmof.XWritableField;
import org.xydra.base.rmof.XWritableObject;
import org.xydra.base.value.XValue;
import org.xydra.index.iterator.Iterators;

/**
 * Wraps an {@link XWritableObject} and turns it into an index from XID -> XID
 * (but a different one).
 *
 * Used for com.calpano.common.server.user.index.EmailIndex
 *
 * IMPROVE 2012-06 future versions might implement IMapIndex<XID,XID>
 *
 * @author xamde
 */
public class XidIndex {

	private final XWritableObject baseObject;

	public XidIndex(final XWritableObject baseObject) {
		this.baseObject = baseObject;
	}

	/**
	 * @param key
	 *            to be looked up
	 * @return the currently indexed XID or null if none found.
	 */
	public XId lookupID(final XId key) {
		final XWritableField indexField = this.baseObject.getField(key);
		if (indexField == null) {
			return null;
		}
		final XId indexValue = (XId) indexField.getValue();
		assert indexValue != null : indexField.getValue();
		return indexValue;
	}

	/**
	 * @param key
	 *            to be indexed
	 * @param value
	 *            to be indexed
	 * @return the previously indexed value, if any, or null.
	 */
	public XId index(final XId key, final XId value) {
		if (key == null) {
			throw new IllegalArgumentException("key may not be null");
		}
		if (value == null) {
			throw new IllegalArgumentException("value may not be null");
		}
		XWritableField indexField = this.baseObject.getField(key);
		if (indexField == null) {
			indexField = this.baseObject.createField(key);
		}
		final XValue xvalue = indexField.getValue();
		final XId indexValue = (XId) xvalue;
		XId previous;
		if (indexValue == null) {
			previous = null;
		} else {
			previous = indexValue;
		}
		indexField.setValue(value);
		return previous;
	}

	/**
	 * @param key
	 *            never null
	 * @return true if index contains the given key
	 */
	public boolean contains(final XId key) {
		assert key != null;
		return this.baseObject.hasField(key);
	}

	/**
	 * @param key
	 *            to de-index
	 */
	public void deindex(final XId key) {
		assert key != null;
		this.baseObject.removeField(key);
	}

	public void clear() {
		final Collection<XId> ids = Iterators.addAll(this.baseObject.iterator(), new HashSet<XId>());
		for (final XId id : ids) {
			this.baseObject.removeField(id);
		}
	}

	public Iterator<XId> keyIterator() {
		return this.baseObject.iterator();
	}

}
