package com.calpano.common.shared.xydrautils.field;

import java.util.HashSet;
import java.util.Set;

import org.xydra.base.BaseRuntime;
import org.xydra.base.XId;
import org.xydra.base.rmof.XWritableField;
import org.xydra.base.rmof.XWritableObject;
import org.xydra.base.value.XStringSetValue;
import org.xydra.base.value.XV;

/**
 * A {@link FieldProperty} managing a {@link XStringSetValue} as a java
 * {@link Set} of {@link String}.
 *
 * @author xamde
 */
public class StringSetFieldProperty extends ExtensibleFieldProperty<Set<String>, XStringSetValue> {

	public StringSetFieldProperty(final String fieldIdString) {
		super(fieldIdString, new Converter<Set<String>, XStringSetValue>() {

			@Override
			public Set<String> toJavaType(final XStringSetValue xydraType) {
				if (xydraType == null) {
					return new HashSet<String>();
				} else {
					return xydraType.toSet();
				}
			}

			@Override
			public XStringSetValue toXydraValue(final Set<String> anyType) {
				if (anyType == null) {
					return BaseRuntime.getValueFactory().createStringSetValue(new String[] {});
				}
				return BaseRuntime.getValueFactory().createStringSetValue(anyType);
			}
		});
	}

	public void addToCollection(final XId actorId, final XWritableObject xo, final String tag) {
		final XStringSetValue oldValue = getXValue(xo);
		XStringSetValue newValue;
		if (oldValue == null) {
			newValue = XV.toStringSetValue(new String[] { tag });
		} else {
			newValue = oldValue.add(tag);
		}
		final XWritableField xf = xo.createField(this.fieldId);
		xf.setValue(newValue);
	}

	public void removeFromCollection(final XId actorId, final XWritableObject xo, final String tag) {
		final XStringSetValue oldValue = getXValue(xo);
		if (oldValue == null) {
			// nothing to remove from
			return;
		} else {
			final XStringSetValue newValue = oldValue.remove(tag);
			if (newValue.size() < oldValue.size()) {
				final XWritableField xf = xo.getField(this.fieldId);
				assert xf != null;
				xf.setValue(newValue);
			}
		}
	}

	private static final XStringSetValue EMPTY = XV.toStringSetValue(new String[] {});

	public void clear(final XWritableObject xo) {
		final XWritableField xf = xo.createField(this.fieldId);
		xf.setValue(EMPTY);
	}

}
