package de.xam.cds;

/**
 * List of CDS relation names in technical form and as human-readable labels.
 *
 * Defines the human-readable standard labels.
 *
 * See page 148 pp in "Personal Knowledge Models with Semantic Models",
 * available from http://pkm.xam.de
 *
 * @author xamde
 */
public enum CdsId {

	/** === CDS data model */
	hasInverse("has inverse", false, Kind.Relation),

	/** === CDS extension vocabulary */
	// ------ types

	type_relation("relation", false, Kind.Item),

	type_transitiveRelation("transitive relation", false, Kind.Item),

	type_symmetricRelation("symmetric relation", false, Kind.Item),

	/** === CDS relation vocabulary */
	// ====== layer 0
	hasRelated("is related to", false, Kind.Relation),

	// ====== layer 1
	hasTarget("links to", false, Kind.Relation),

	hasSource("is linked from", false, Kind.Relation),

	// symmetric
	hasSimilar("is similar to", false, Kind.Relation),

	// ====== layer 2

	hasAnnotationMember("annotates", false, Kind.Relation),

	// fix typo on 2015-08-30 (leading space removed)
	hasAnnotation("has annotation", false, Kind.Relation),

	hasAfter("comes before", true, Kind.Relation),

	hasBefore("comes after", true, Kind.Relation),

	hasDetail("has detail", false, Kind.Relation),

	hasContext("has context", false, Kind.Relation),

	/* added on 2014-07-13 */
	hasAutoLinkTarget("auto-links to", false, Kind.Relation),

	/* added on 2014-07-13 */
	hasAutoLinkSource("auto-linked from", false, Kind.Relation),

	/* added on 2014-07-13 */
	hasManualLinkTarget("wiki-links to", false, Kind.Relation),

	/* added on 2014-07-13 */
	hasManualLinkSource("wiki-linked from", false, Kind.Relation),

	// symmetric
	sameAs("is same as", true, Kind.Relation),

	replaces("replaces", false, Kind.Relation),

	replacedBy("is replaced by", false, Kind.Relation),

	isAliasOf("is alias for", false, Kind.Relation),

	hasAlias("has alias", false, Kind.Relation),

	// ====== layer 3
	hasTagMember("is tag of", false, Kind.Relation),

	hasTag("is tagged with", false, Kind.Relation),

	/** inverseOf hasSupertype */
	hasSubType("has subType", true, Kind.Relation),

	/** type hierarchy for items and relations */
	hasSuperType("is subtype of", true, Kind.Relation),

	hasPart("has part", false, Kind.Relation),

	isPartOf("is part of", false, Kind.Relation),

	// ====== layer 4
	hasInstance("has instance", false, Kind.Relation),

	/** Similar to rdf:hasType */
	hasType("has type", false, Kind.Relation)

	;

	public static enum Kind {
		Relation, Item
	}

	/**
	 * Axioms about inverse relations, not in triple form
	 */
	public static void initInverses() {
		/** CDS data model layer */
		hasInverse.setInverse(hasInverse);
		/** as defined on page 148pp */
		hasRelated.setInverse(hasRelated);
		hasTarget.setInverse(hasSource);
		hasAnnotationMember.setInverse(hasAnnotation);
		hasTagMember.setInverse(hasTag);
		hasInstance.setInverse(hasType);
		hasAfter.setInverse(hasBefore);
		hasDetail.setInverse(hasContext);
		hasSubType.setInverse(hasSuperType);
		hasPart.setInverse(isPartOf);
		hasTarget.setInverse(hasSource);
		hasSimilar.setInverse(hasSimilar);
		sameAs.setInverse(sameAs);
		isAliasOf.setInverse(hasAlias);
		replaces.setInverse(replacedBy);
		/** end of definitions from page 148pp */
	}

	static {
		initInverses();
	}

	private String label;

	public boolean isRelation() {
		return this.kind == Kind.Relation;
	}

	public boolean isTransitiveRelation() {
		return this.isTransitiveRelation;
	}

	private boolean isTransitiveRelation;

	private Kind kind;

	/**
	 * @return default English human readable label
	 */
	public String getLabel() {
		return this.label;
	}

	/**
	 * @param label
	 * @param isTransitiveRelation
	 * @param kind
	 */
	CdsId(final String label, final boolean isTransitiveRelation, final Kind kind) {
		assert !isTransitiveRelation || kind == Kind.Relation;
		this.label = label;
		this.isTransitiveRelation = isTransitiveRelation;
		this.kind = kind;
	}

	public CdsId getInverse() {
		return this.inverse;
	}

	private CdsId inverse;

	public void setInverse(final CdsId inverse) {
		this.inverse = inverse;
	}

	public boolean isItem() {
		return this.kind == Kind.Item;
	}

}
