package de.xam.googleanalytics.httpclient;

import java.io.IOException;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.Future;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import java.util.concurrent.atomic.AtomicInteger;

import org.apache.commons.httpclient.DefaultHttpMethodRetryHandler;
import org.apache.commons.httpclient.HttpClient;
import org.apache.commons.httpclient.methods.GetMethod;
import org.xydra.log.api.Logger;
import org.xydra.log.api.LoggerFactory;

/**
 * TODO fix timeout handling future vs. httpclient
 *
 * @author xamde
 *
 */
public class HttpUserAgentApacheCommons implements HttpUserAgent {

	private static final Logger log = LoggerFactory.getLogger(HttpUserAgentApacheCommons.class);

	private final HttpClient httpClient;

	private final LinkedBlockingQueue<Job> jobQueue;

	private TrackingThread workerThread;

	public HttpUserAgentApacheCommons() {
		super();
		this.httpClient = new HttpClient();
		this.jobQueue = new LinkedBlockingQueue<Job>(1000);
	}

	@Override
	public void setUserAgentIdentifier(final String userAgent) {
		System.getProperties().setProperty("httpclient.useragent", userAgent);
	}

	@Override
	public void setConnectionTimeout(final int maxMillis) {
		this.httpClient.getHttpConnectionManager().getParams().setConnectionTimeout(5000);
	}

	@Override
	public void setAutoRetry(final boolean autoRetry) {
		this.httpClient
				.getHttpConnectionManager()
				.getParams()
				.setParameter("http.method.retry-handler",
						new DefaultHttpMethodRetryHandler(0, false));
	}

	AtomicInteger attempts = new AtomicInteger();
	AtomicInteger successes = new AtomicInteger();
	AtomicInteger faillures = new AtomicInteger();

	private class TrackingThread extends Thread {

		private boolean shouldRun;

		public TrackingThread() {
			setPriority(Thread.MIN_PRIORITY);
		}

		@Override
		public void run() {
			HttpUserAgentApacheCommons.this.attempts.incrementAndGet();

			this.shouldRun = true;
			giveUpOnBadNetwork();

			while (this.shouldRun) {
				try {
					final Job job = HttpUserAgentApacheCommons.this.jobQueue.take();
					int status = HttpUserAgentApacheCommons.httpGet(
							HttpUserAgentApacheCommons.this.httpClient, job);
					// retry every 20 seconds
					while (!isOkStatus(status) && this.shouldRun) {
						log.warn("HTTP GET failed, retry in 20 seconds");
						HttpUserAgentApacheCommons.this.faillures.incrementAndGet();
						Thread.sleep(20000);
						status = HttpUserAgentApacheCommons.httpGet(
								HttpUserAgentApacheCommons.this.httpClient, job);
						giveUpOnBadNetwork();
					}
					HttpUserAgentApacheCommons.this.successes.incrementAndGet();
				} catch (final InterruptedException e) {
					log.debug("waiting for urls to track", e);
				}
			}
		}

		private void giveUpOnBadNetwork() {
			if (HttpUserAgentApacheCommons.this.successes.get() == 0
					&& HttpUserAgentApacheCommons.this.faillures.get() >= 3
					&& HttpUserAgentApacheCommons.this.attempts.get() < 50) {
				this.shouldRun = false;
				log.warn("So many HTTP errors, not even trying to GET");
			} else {
				log.info("HTTP STATUS: " + HttpUserAgentApacheCommons.this.attempts.get()
						+ " attempts: " + HttpUserAgentApacheCommons.this.successes.get()
						+ " succ with " + HttpUserAgentApacheCommons.this.faillures.get()
						+ " fail ");
			}
		}
	}

	private static boolean isOkStatus(final int status) {
		return 200 <= status && status < 300;
	}

	private static int httpGet(final HttpClient httpClient, final Job job) {
		log.debug("GET: " + job.url);
		final GetMethod get = new GetMethod(job.url);
		try {
			final int status = httpClient.executeMethod(get);
			job.status = status;
			if (isOkStatus(status)) {
				log.trace("Status " + status);
			} else {
				log.debug("Status code is " + status);
			}
			return status;
		} catch (final IOException e) {
			log.info("Network error. Could not track " + job.url);
			log.debug("Network error = ", e);
			return 404;
		} finally {
			get.releaseConnection();
		}

	}

	private static final long toMillis(final long time, final TimeUnit unit) {
		final long result = time;
		switch (unit) {
		case NANOSECONDS:
			return result / 1000000;
		case MICROSECONDS:
			return result / 1000;
		case MILLISECONDS:
			return result;
		case SECONDS:
			return result * 1000;
		case MINUTES:
			return result * 1000 * 60;
		case HOURS:
			return result * 1000 * 60 * 60;
		case DAYS:
			return result * 1000 * 60 * 60 * 24;
		}
		throw new AssertionError();
	}

	private static class Job implements Future<Integer> {
		public int status = -1;
		String url;
		private boolean cancelled = false;
		private final boolean done = false;

		public Job(final String url) {
			this.url = url;
		}

		@Override
		public boolean cancel(final boolean mayInterruptIfRunning) {
			if (this.done || this.cancelled) {
				return false;
			} else {
				this.cancelled = true;
				return true;
			}
		}

		@Override
		public Integer get() throws InterruptedException, ExecutionException {
			while (!isDone() && !isCancelled() && this.status < 0) {
				Thread.sleep(1000);
			}
			return this.status;
		}

		@Override
		public Integer get(final long timeout, final TimeUnit unit) throws InterruptedException,
				ExecutionException, TimeoutException {
			while (!isDone() && !isCancelled() && this.status < 0) {
				Thread.sleep(toMillis(timeout, unit));
			}
			return null;
		}

		@Override
		public boolean isCancelled() {
			return this.cancelled;
		}

		@Override
		public boolean isDone() {
			return this.done;
		}
	}

	@Override
	public synchronized Future<Integer> GET(final String url) {
		// put in queue
		final Job job = new Job(url);
		final boolean scheduled = this.jobQueue.offer(job);
		if (!scheduled) {
			log.warn("More than 1000 urls scheduled for tracking.");
		}

		// make sure the worker thread runs
		if (this.workerThread == null) {
			this.workerThread = new TrackingThread();
			this.workerThread.start();
		}

		return job;
	}

}
