package de.xam.googleanalytics.logsink;

import java.net.InetAddress;
import java.net.UnknownHostException;
import java.util.HashMap;
import java.util.Map;

import org.xydra.log.api.ILogListener;
import org.xydra.log.api.Logger;

import de.xam.googleanalytics.FocusPoint;
import de.xam.googleanalytics.GaEvent;
import de.xam.googleanalytics.Tracker;
import de.xam.googleanalytics.UrlAndHashUtils;
import de.xam.googleanalytics.UserInfo;

/**
 * Routes Xydra log events (of level 'warn' and 'error') to Google Analytics
 * events.
 *
 * To avoid endless loops, this class uses no Logger itself.
 *
 * <h3>Syntax</h3> Syntax for data logging via GA?key=val&...
 *
 * Recognised key names are: 'category', 'action', 'label', and 'value'.
 * Category and action are mandatory.
 *
 * Level debug is NEVER logging anything.
 *
 * @author xamde
 */
public class GALogListener implements ILogListener, UserInfo {

	private final long currentSessionStartTime = UrlAndHashUtils.getCurrentTimeInSeconds();

	private final String domainName;

	private final long thirtyOneBitId = UrlAndHashUtils.random31bitInteger();
	private final Tracker tracker;

	/**
	 * @param trackerCode
	 *            like 'UA-271022-28'
	 * @param domainName
	 *            without leading www. Other 3rd level domain names may remain.
	 */
	public GALogListener(final String trackerCode, final String domainName) {
		this.tracker = new Tracker(trackerCode);
		this.domainName = domainName;
	}

	@Override
	public void debug(final Logger log, final String msg) {
		if (hasGaData(msg)) {
			trackData(log, "info", msg);
		}
	}

	@Override
	public void debug(final Logger log, final String msg, final Throwable t) {
		if (hasGaData(msg)) {
			trackData(log, "info", msg);
		}
	}

	@Override
	public void error(final Logger log, final String msg) {
		if (hasGaData(msg)) {
			trackData(log, "error", msg);
		}
	}

	@Override
	public void error(final Logger log, final String msg, final Throwable t) {
		if (hasGaData(msg)) {
			trackData(log, "error", msg);
		}
	}

	@Override
	public long get31BitId() {
		return this.thirtyOneBitId;
	}

	@Override
	public long getCurrentSessionStartTime() {
		return this.currentSessionStartTime;
	}

	@Override
	public String getDomainName() {
		return this.domainName;
	}

	@Override
	public long getFirstVisitStartTime() {
		return this.currentSessionStartTime;
	}

	@Override
	public String getHostName() {
		InetAddress addr;
		String hostname = "(not set)";
		try {
			addr = InetAddress.getLocalHost();
			hostname = addr.getHostName();
		} catch (final UnknownHostException e) {
			// ("Could not determine local hostname, never mind", e);
		}
		return hostname;
	}

	@Override
	public long getLastVisitStartTime() {
		return this.currentSessionStartTime;
	}

	@Override
	public long getSessionCount() {
		return 1;
	}

	public Tracker getTracker() {
		return this.tracker;
	}

	// TODO document what would be a legal value and re-enable in UrchinCookie
	// code
	@Override
	public String getVar() {
		return null;
	}

	@Override
	public void info(final Logger log, final String msg) {
		if (hasGaData(msg)) {
			trackData(log, "info", msg);
		}
	}

	@Override
	public void info(final Logger log, final String msg, final Throwable t) {
		if (hasGaData(msg)) {
			trackData(log, "info", msg);
		}
	}

	@Override
	public void trace(final Logger log, final String msg) {
		if (hasGaData(msg)) {
			trackData(log, "info", msg);
		}
	}

	@Override
	public void trace(final Logger log, final String msg, final Throwable t) {
		if (hasGaData(msg)) {
			trackData(log, "info", msg);
		}
	}

	/**
	 * @param log
	 * @param gaEvent
	 */
	private void track(final Logger log, final GaEvent gaEvent) {
		this.tracker.track(new FocusPoint(log.toString()), "-", this, gaEvent);
	}

	/**
	 * @param log
	 *            ..
	 * @param logLevel
	 *            ..
	 * @param msg
	 *            must have format 'bla bla bla GA?key=val?key=val' (end)
	 */
	private void trackData(final Logger log, final String logLevel, final String msg) {
		assert hasGaData(msg);
		final String dataPart = msg.substring(msg.indexOf(GA_URI) + GA_URI.length());
		final Map<String, String> dataMap = parseQueryString(dataPart);
		GaEvent gaEvent;
		final String category = dataMap.get("category");
		assert category != null;
		final String action = dataMap.get("action");
		assert action != null;
		final String optionalLabel = dataMap.get("label");
		final String optionalValue = dataMap.get("value");
		int value = -1;
		if (optionalValue != null) {
			value = Integer.parseInt(optionalValue);
		}
		if (optionalLabel == null) {
			gaEvent = new GaEvent(category, action);
		} else {
			if (value == -1) {
				gaEvent = new GaEvent(category, action, optionalLabel);
			} else {
				gaEvent = new GaEvent(category, action, optionalLabel, value);
			}
		}

		track(log, gaEvent);
	}

	@Override
	public void warn(final Logger log, final String msg) {
		if (hasGaData(msg)) {
			trackData(log, "warn", msg);
		}
	}

	@Override
	public void warn(final Logger log, final String msg, final Throwable t) {
		if (hasGaData(msg)) {
			trackData(log, "warn", msg);
		}
	}

	public static final String GA_URI = "GA?";

	public static boolean hasGaData(final String s) {
		return s.contains(GA_URI);
	}

	public static Map<String, String> parseQueryString(final String q) {
		final Map<String, String> map = new HashMap<String, String>();
		final String[] pairs = q.split("\\&");

		for (final String s : pairs) {
			final String[] parts = s.split("=");
			final String key = parts[0];
			final String value = parts.length > 1 ? parts[1] : null;
			map.put(key, value);
		}
		return map;
	}

}
