package de.xam.p13n.shared;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;
import java.util.Set;
import java.util.Stack;

import org.xydra.annotations.Feature;
import org.xydra.annotations.RequiresAppEngine;
import org.xydra.annotations.RunsInAppEngine;
import org.xydra.annotations.RunsInGWT;
import org.xydra.base.minio.MiniBufferedReader;
import org.xydra.base.minio.MiniIOException;
import org.xydra.base.minio.MiniReader;
import org.xydra.base.minio.MiniStringReader;
import org.xydra.log.api.Logger;
import org.xydra.log.api.LoggerFactory;
import org.xydra.sharedutils.XyAssert;

/**
 * A velocity context that returns values based on a given
 * {@link Personalisation}.
 *
 * @author xamde
 */
@RunsInGWT(true)
@RunsInAppEngine(true)
@RequiresAppEngine(false)
public class PersonalisedMessageBundle implements Map<String, String>, Serializable {

	private static Logger log = LoggerFactory.getLogger(PersonalisedMessageBundle.class);

	public static final String MSG_FILE_EXTENSION = "utf8";

	public static final String MSG_FILE_NAME_PREFIX = "MSG";

	private static final long serialVersionUID = -5677588162410333376L;

	private static final String KEY_P13N = "_p13n";

	public static final String KEY_VERSIONHASH = "_version";

	public static final String KEY_TYPE = "_type";

	/**
	 * If true, missing keys are rendered as '???( messageKeyName )'
	 */
	public static boolean SHOW_MISSING_KEYS_IN_UI = true;

	/**
	 * See
	 * {@link P13nSearchUtils#getFromGeneralToSpecific(String, Personalisation, String, String)}
	 *
	 * @param packageName
	 *            where to search
	 * @param p13n
	 *            may be null
	 * @return an ordered Stack of resource path names
	 */
	public static Stack<String> getFromGeneralToSpecific(final String packageName, final Personalisation p13n) {
		return P13nSearchUtils.getFromGeneralToSpecific(packageName, p13n, MSG_FILE_NAME_PREFIX,
				MSG_FILE_EXTENSION);
	}

	/**
	 * @param classOrPackageName
	 *            in slash-notation
	 * @return the parent-folder name of the given classOrPackageName
	 */
	public static String moveOneFolderUp(final String classOrPackageName) {
		if (!classOrPackageName.contains("/")) {
			throw new IllegalArgumentException("'" + classOrPackageName
					+ "' does not contain a '/'");
		}
		final int last = classOrPackageName.lastIndexOf('/');
		return classOrPackageName.substring(0, last);
	}

	private Map<String, String> map = new HashMap<String, String>();

	private transient Personalisation p13n;

	private transient String[] packageNames;

	private transient String hash;

	/* Just for GWT */
	protected PersonalisedMessageBundle() {
	}

	/**
	 * @param p13n
	 *            for debugging purposes
	 * @param packageNames
	 *            for debugging purposes
	 */
	public PersonalisedMessageBundle(final Personalisation p13n, final String... packageNames) {
		assert p13n != null;
		this.p13n = p13n;
		this.map.put(KEY_P13N, p13n.toCompactString());
		this.packageNames = packageNames;
	}

	@Override
	public void clear() {
		this.map.clear();
	}

	public Iterator<String> getRawKeys() {
		return this.map.keySet().iterator();
	}

	@Override
	public boolean containsKey(final Object key) {
		return this.map.containsKey(key);
	}

	@Override
	public boolean containsValue(final Object value) {
		return this.map.containsValue(value);
	}

	@Override
	public Set<java.util.Map.Entry<String, String>> entrySet() {
		return this.map.entrySet();
	}

	@Override
	public boolean equals(final Object other) {
		if (!(other instanceof PersonalisedMessageBundle)) {
			return false;
		}
		final PersonalisedMessageBundle otherPMB = (PersonalisedMessageBundle) other;
		return this.map.equals(otherPMB.map);
	}

	public String getHash() {
		if (this.hash == null) {
			this.hash = "" + this.map.hashCode();
		}
		return this.hash;
	}

	@Override
	public String get(final Object keyObject) {
		String value = null;
		final String key = (String) keyObject;

		if (this.map.containsKey(key)) {
			// key known, but maybe set to null explicitly
			value = this.map.get(key);
		} else {
			// key completely unknown
			MissingMessageKeys.put(key, this);
			log.warn("Missing message key '"
					+ key
					+ "' requested. Stop your debugger here and inspect the this.uber object of Velocity.");
			if (SHOW_MISSING_KEYS_IN_UI) {
				return "???(" + key + ")";
			}
		}

		// debug
		if (SharedP13nUtils.isDebugger(getPersonalisation())) {
			value = debugMode(key, value);
		}

		// post-process
		if (value != null) {
			if (SharedP13nUtils.isLoremIpsum(getPersonalisation())) {
				value = SharedP13nUtils.loremIpsum(value);
			}
		}
		return value;
	}

	private static String debugMode(final String key, final String value) {
		if (value == null) {
			return "[" + key + "(missing)]";
		} else {
			return value + " [" + key + "]";
		}
	}

	public Object[] getKeys() {
		log.warn("ALARM! A call to getKeys. The result is incomplete and contains only those keys added by previous calls.");
		final ArrayList<String> list = new ArrayList<String>(this.map.keySet());
		return list.toArray(new String[this.map.keySet().size()]);
		// throw new
		// UnsupportedOperationException("I thought velocity is never using this");
	}

	@Override
	public int hashCode() {
		return this.map.hashCode() + this.p13n.hashCode();
	}

	@Override
	public boolean isEmpty() {
		return this.map.isEmpty();
	}

	@Override
	public Set<String> keySet() {
		return this.map.keySet();
	}

	@Override
	public String put(final String arg0, final String arg1) {
		return this.map.put(arg0, arg1);
	}

	@Override
	public void putAll(final Map<? extends String, ? extends String> arg0) {
		this.map.putAll(arg0);
	}

	@Override
	public String remove(final Object arg0) {
		return this.map.remove(arg0);
	}

	@Override
	public int size() {
		return this.map.size();
	}

	@Override
	public String toString() {
		return this.p13n.toClassifier((short) 3) + " in packages " + toString(this.packageNames);
	}

	private static String toString(final String[] packageNames) {
		if (packageNames == null) {
			return "[]";
		}
		final StringBuffer buf = new StringBuffer("[");
		for (final String pn : packageNames) {
			buf.append(pn).append(", ");
		}
		buf.append("]");
		return buf.toString();
	}

	@Override
	public Collection<String> values() {
		return this.map.values();
	}

	public Personalisation getPersonalisation() {
		if (this.p13n == null) {
			final String p13nCompactString = this.map.get(KEY_P13N);
			log.info("p13n from " + p13nCompactString);
			assert p13nCompactString != null;
			this.p13n = Personalisation.fromString(p13nCompactString);
		}
		return this.p13n;
	}

	public String toStorageString() {
		final StringBuilder sb = new StringBuilder();
		for (final Map.Entry<String, String> e : this.map.entrySet()) {
			final String key = e.getKey();
			XyAssert.xyAssert(key != null);
			assert key != null;
			XyAssert.xyAssert(!key.equals(""));
			String value = e.getValue();
			/*
			 * When serialising a "key=(empty string)" the serialisation should
			 * not become "key=null", which is later a problem.
			 */
			if (value == null) {
				value = "";
			}
			value = value.replace("\n", "\\n");
			value = value.replace("\r", "\\r");
			sb.append(key + "=" + value + "\n");
		}
		return sb.toString();
	}

	public static PersonalisedMessageBundle fromStorageString(final String s) {
		final MiniStringReader msr = new MiniStringReader(s);
		final Map<String, String> map = toMap(msr, "localStorage");
		final PersonalisedMessageBundle bundle = new PersonalisedMessageBundle();
		bundle.map = map;
		return bundle;
	}

	private static class ParseContext {
		String key;
		StringBuilder valueBuf;
		boolean addNewline;

		public ParseContext() {
			reset();
		}

		public void parseValue(final String value) {
			boolean addNextNewline = true;

			/* handle white space-only-values */
			String v = value;
			final int i = v.indexOf("##");
			if (i >= 0) {
				addNextNewline = false;
				// no trim
				v = v.substring(0, i);
			}

			/* expand inline line breaks */
			v = v.replace("\\n", "\n");

			// append a newline first
			if (this.addNewline) {
				this.valueBuf.append("\n");
			}
			this.valueBuf.append(v);
			this.addNewline = addNextNewline;
		}

		public void flushTo(final Map<String, String> map, final int lineNo, final String source) {
			if (this.key != null) {
				final String value = this.valueBuf.toString();
				/*
				 * empty value, store as null to distinguish from missing keys
				 * and still render $varname in velocity to find errors
				 */
				if (map.containsKey(this.key)) {
					throw new IllegalArgumentException("Key '" + this.key
							+ "' was already defined. Just parsing " + source + ":" + lineNo);
				}
				map.put(this.key, value.length() == 0 ? null : value);
				log.trace("Mapping " + this.key + " = '" + value + "'");
				reset();
			}
		}

		private void reset() {
			this.key = null;
			this.valueBuf = new StringBuilder();
			// no initial newline
			this.addNewline = false;
		}
	}

	/**
	 * Extended properties file syntax. Lines starting with '#+' are interpreted
	 * as continuing a value started on the line before.
	 *
	 * @param reader
	 * @param source
	 * @return ...
	 * @throws MiniIOException
	 */
	@Feature("parsing")
	public static Map<String, String> toMap(final MiniReader reader, final String source)
			throws MiniIOException {
		final Map<String, String> map = new HashMap<String, String>();
		final MiniBufferedReader br = new MiniBufferedReader(reader);

		// accepts \n, \r, or \r\n
		String line = br.readLine();
		int lineNo = 1;
		final ParseContext parseContext = new ParseContext();
		while (line != null) {
			line = line.trim();
			// classify line
			if (line.startsWith("#+")) {
				// line continues a multi-line value
				parseContext.parseValue(line.substring(2));
			} else if (line.startsWith("#") || line.equals("")) {
				// line is a comment or empty, skip completely
			} else {
				// normal line, must contains '='
				final int pos = line.indexOf('=');
				assert pos > 0 : "parsing line " + lineNo + " in " + source
						+ " failed. Found no '=' sign. Line = '" + line + "'";
				// emit last key, if defined
				parseContext.flushTo(map, lineNo, source);
				parseContext.key = line.substring(0, pos);

				if (parseContext.key.equals("")) {
					log.warn("Suspicious line nr. " + lineNo + ": " + line);
				}

				if (pos + 1 == line.length()) {
					// nothing to be added to value
				} else {
					final String value = line.substring(pos + 1, line.length());
					/* handle white space-only-values */
					if (value.trim().length() == 0) {
						// nothing to be added to value
					} else {
						parseContext.parseValue(value.trim());
					}
				}
			}
			lineNo++;
			line = br.readLine();
		}
		parseContext.flushTo(map, lineNo, source);
		return map;
	}
}
