package de.xam.texthtml.text;

import java.text.DecimalFormat;
import java.text.DecimalFormatSymbols;
import java.text.NumberFormat;
import java.util.Locale;

import org.joda.time.DateTime;
import org.joda.time.DateTimeZone;
import org.joda.time.Days;
import org.joda.time.Interval;
import org.joda.time.LocalDate;
import org.joda.time.chrono.ISOChronology;
import org.joda.time.format.DateTimeFormat;
import org.joda.time.format.DateTimeFormatter;
import org.joda.time.format.ISODateTimeFormat;
import org.xydra.annotations.RequiresAppEngine;
import org.xydra.annotations.RunsInAppEngine;
import org.xydra.annotations.RunsInGWT;
import org.xydra.log.api.Logger;
import org.xydra.log.api.LoggerFactory;

import de.xam.p13n.server.LocaleUtils;
import de.xam.p13n.shared.Personalisation;
import de.xam.p13n.shared.PersonalisedMessageBundle;
import de.xam.p13n.shared.time.TimeProvider;
import de.xam.resourceloader.p13n.P13nUtils;
import de.xam.resourceloader.p13n.PersonalisedMessageBundleIO;

/**
 * Formats technical values into nice human readable ones (e.g. time values).
 *
 * Uses joda time and java locale.
 *
 * Many methods required a {@link Personalisation}.
 *
 * This class is designed to be used in velocity templates.
 *
 * @author xamde
 *
 */
@RunsInGWT(false)
@RunsInAppEngine(true)
@RequiresAppEngine(false)
public class HumanReadableText {

	private static final Logger log = LoggerFactory.getLogger(HumanReadableText.class);

	public static final long MILLIS_PER_MINUTE = 60 * 1000;
	public static final long MILLIS_PER_HOUR = 60 * MILLIS_PER_MINUTE;
	public static final long MILLIS_PER_DAY = 24 * MILLIS_PER_HOUR;

	/**
	 * Return e.g. 2014-12-31--14-55-56,999
	 *
	 * @param datetime ..
	 * @return a debug-oriented nicely readable non-standard date time
	 */
	public static String fullDateTime(final long datetime) {
		final DateTime dt = new DateTime(datetime, DateTimeZone.UTC);
		return dt.toString("yyyy'-'MM'-'dd'--'HH'-'mm'-'ss','SSS'-Z'ZZZZZ");
	}

	/**
	 * Return e.g. 2014-12-31--14-55 -- a good format, e.g. for software
	 * releases
	 *
	 * @param datetime ..
	 * @return a debug-oriented nicely readable non-standard date time
	 */
	public static String fullDate_HH_MM(final long datetime) {
		final DateTime dt = new DateTime(datetime, DateTimeZone.UTC);
		return dt.toString("yyyy'-'MM'-'dd'--'HH'-'mm'");
	}

	/**
	 * @param datetime
	 * @return given datetime as 'yyyy-MM-dd'
	 */
	public static String fullDate(final long datetime) {
		if (datetime == 0) {
			return "-";
		}

		final DateTime dt = new DateTime(datetime, DateTimeZone.UTC);
		return dt.toString("yyyy'-'MM'-'dd'");
	}

	/**
	 * @param datetime
	 * @return '2015-07-20T12:43:45.059Z'
	 */
	public static String fullDateTimeIso(final long datetime) {
		if (datetime == 0) {
			return "-";
		}

		final DateTime dt = new DateTime(datetime, DateTimeZone.UTC);
		return dt.toString(ISODateTimeFormat.dateTime());
	}

	public static long parseIsoDatetime(final String fullDateTimeIso) {
		final DateTimeFormatter dtf = ISODateTimeFormat.dateTimeParser();
		final DateTime dt = dtf.parseDateTime(fullDateTimeIso);
		return dt.getMillis();
	}

	/**
	 * Great for cohort anlysis
	 *
	 * @param date a date after start
	 * @param start a start date
	 * @return number of weeks that date is after start
	 */
	public static int weeksSinceStart(final long date, final long start) {
		return new Interval(start, date, ISOChronology.getInstanceUTC()).toPeriod().getWeeks();
	}

	/**
	 * @param datetime UTC
	 * @param p13n never null
	 * @return a full ISO8601 date and time String personalized with the given
	 *         {@link Personalisation}
	 */
	public static String fullDateTime(final long datetime, final Personalisation p13n) {
		final int offset = p13n.getUtcOffset();
		final DateTimeZone p13nZimeZone = DateTimeZone.forOffsetHours(offset);
		final DateTime dt = new DateTime(datetime, p13nZimeZone);
		return dt.toString("yyyy'-'MM'-'dd'--'HH'-'mm'-'ss','SSS'-Z'ZZZZZ");
	}

	/**
	 * @param datetime UTC
	 * @param p13n never null
	 * @return A string with date and time as expected by users with the given
	 *         {@link Personalisation}
	 */
	public static String commonDateTime(final long datetime, final Personalisation p13n) {
		final int offset = p13n.getUtcOffset();
		final DateTimeZone p13nZimeZone = DateTimeZone.forOffsetHours(offset);
		final DateTime dt = new DateTime(datetime, p13nZimeZone);
		final Locale javaLocale = LocaleUtils.toJavaLocale(p13n.getLocale());
		final DateTimeFormatter dtf = DateTimeFormat.mediumDateTime().withLocale(javaLocale);
		return dtf.print(dt);
	}

	/**
	 * @param now UTC
	 * @param datetime UTC
	 * @return true if datetime is on same day as now
	 */
	public static boolean isToday(final long now, final long datetime) {
		final LocalDate _now = new LocalDate(now);
		final LocalDate _datetime = new LocalDate(datetime);
		return _datetime.isEqual(_now);
	}

	/**
	 * Relies on {@link Personalisation}
	 *
	 * @param now in UTC
	 * @param datetime in user time zone
	 * @param p13n never null
	 * @return a nicely formatted String such as "Heute" or "Freitag, 3. Mai"
	 */
	public static String niceDay(final long now, final long datetime, final Personalisation p13n) {
		if (p13n == null) {
			return "p13n missing";
		}

		final int offset = p13n.getUtcOffset();
		final DateTimeZone p13nZimeZone = DateTimeZone.forOffsetHours(offset);
		final DateTime dt = new DateTime(datetime, p13nZimeZone);

		final LocalDate _now = new LocalDate(now, p13nZimeZone);
		final LocalDate _datetime = dt.toLocalDate();

		log.debug("Difference in hours is: " + (now - dt.getMillis()) / (1000 * 60 * 60));
		log.debug("Now: " + _now.toString() + " Date: " + _datetime.toString());

		// handling different locales
		final PersonalisedMessageBundle msg = PersonalisedMessageBundleIO.instance(p13n);

		final Locale locale = P13nUtils.toJavaLocale(p13n.getLocale());
		if (_datetime.isEqual(_now)) {
			return msg.get("time-today");
		} else if (_datetime.plusDays(1).equals(_now)) {
			return msg.get("time-yesterday");
		} else if (_datetime.plusDays(2).equals(_now)) {
			return msg.get("time-beforeYesterday");
		} else if (_datetime.minusDays(1).equals(_now)) {
			return msg.get("time-tomorrow");
		} else // if within +/- 90 days:
			if (Days.daysBetween(_now, _datetime).getDays() <= 90) {
				// "Freitag, 3. Mai"
				final String weekday = _datetime.dayOfWeek().getAsText(locale);
				final String dayAndMonth = _datetime.toString("d'. 'MMMM", locale);
				return weekday + ", " + dayAndMonth;
			} else
				// 3. Mai 2008"
			{
				return _datetime.toString("d'. 'MMMM' 'yyyy", locale);
			}
	}

	/**
	 * TODO 2012-02 P13n: use joda time here
	 *
	 * @param milliseconds ..
	 * @return a nicely formatted string
	 */
	public static String niceTimespan(final long milliseconds) {
		final long secsTotal = milliseconds / 1000;
		final long minsTotal = secsTotal / 60;
		final long hoursTotal = minsTotal / 60;
		final long daysTotal = hoursTotal / 24;
		final long weeksTotal = daysTotal / 7;
		// months are more tricky
		// long monthTotal = daysTotal / 30;
		final long yearsTotal = daysTotal / 365;

		// long months = monthTotal - (12 * yearsTotal);
		final long weeks = weeksTotal - yearsTotal * 52;
		final long days = daysTotal - weeksTotal * 7;

		final long hours = hoursTotal - daysTotal * 24;
		final long mins = minsTotal - hoursTotal * 60;
		final long secs = secsTotal - minsTotal * 60;

		final StringBuffer buf = new StringBuffer();
		boolean needComma = false;
		int scalesShown = 0;
		final int showScalesMax = 2;

		if (yearsTotal > 0) {
			buf.append(yearsTotal + " year" + (yearsTotal > 1 ? "s" : ""));
			needComma = true;
			scalesShown++;
		}
		// boolean showMonths = months > 0;
		// if(showMonths) {
		// if(needComma) {
		// buf.append(", ");
		// }
		// buf.append(months + " month" + (months > 1 ? "s" : ""));
		// needComma = true;
		// }
		if (weeks > 0 && scalesShown < showScalesMax) {
			if (needComma) {
				buf.append(", ");
			}
			buf.append(weeks + " week" + (weeks > 1 ? "s" : ""));
			needComma = true;
			scalesShown++;
		}
		if (days > 0 && scalesShown < showScalesMax) {
			if (needComma) {
				buf.append(", ");
			}
			buf.append(days + " day" + (days > 1 ? "s" : ""));
			needComma = true;
			scalesShown++;
		}
		if (hours > 0 && scalesShown < showScalesMax) {
			if (needComma) {
				buf.append(", ");
			}
			buf.append(hours + " hour" + (hours > 1 ? "s" : ""));
			needComma = true;
			scalesShown++;
		}
		if (mins > 0 && scalesShown < showScalesMax) {
			if (needComma) {
				buf.append(", ");
			}
			buf.append(mins + " min" + (mins > 1 ? "s" : ""));
			needComma = true;
			scalesShown++;
		}
		if (secs > 0 && scalesShown < showScalesMax) {
			if (needComma) {
				buf.append(", ");
			}
			buf.append(secs + " sec" + (secs > 1 ? "s" : ""));
			needComma = true;
			scalesShown++;
		}

		if (buf.length() == 0) {
			buf.append("instant");
		}
		return buf.toString();
	}

	public static void main(final String[] args) {
		final Personalisation p13n = Personalisation.GERMANY;
		System.out.println("GERMANY = "
				+ HumanReadableText.commonDateTime(TimeProvider.getCurrentTimeInMillis(), p13n));

		System.out.println("EN_US__SAN_FRANCISCO = "
				+ HumanReadableText.commonDateTime(TimeProvider.getCurrentTimeInMillis(),
						Personalisation.EN_US__SAN_FRANCISCO));

		System.out.println(niceTimespan(1000 * 1000 * 1000));

		System.out.println(weeksSinceStart(TimeProvider.getCurrentTimeInMillis() + 2000000000,
				TimeProvider.getCurrentTimeInMillis()));

		final long now = TimeProvider.getCurrentTimeInMillis();
		System.out.println(fullDateTime(now));
		System.out.println("now = " + now + " =?= " + parseIsoDatetime(fullDateTimeIso(now)));
		System.out.println(parseIsoDatetime("2011-10-11T19:00:00.000Z"));
		System.out.println(largeNumber(10000000));
	}

	/**
	 * @param t
	 * @return How many days have passed since t
	 */
	public static int daysSince(final long t) {
		// TODO use FavrSharedSettings here
		final long now = System.currentTimeMillis();
		final long ago = now - t;
		final int days = (int) (ago / MILLIS_PER_DAY);
		return days;
	}

	/*
	 * Symbol Meaning Presentation Examples
	 *
	 * ------ ------- ------------ -------
	 *
	 * G era text AD
	 *
	 * C century of era (>=0) number 20
	 *
	 * Y year of era (>=0) year 1996
	 *
	 * x weekyear year 1996
	 *
	 * w week of weekyear number 27
	 *
	 * e day of week number 2
	 *
	 * E day of week text Tuesday; Tue
	 *
	 * y year year 1996
	 *
	 * D day of year number 189
	 *
	 * M month of year month July; Jul; 07
	 *
	 * d day of month number 10
	 *
	 * a halfday of day text PM
	 *
	 * K hour of halfday (0~11) number 0
	 *
	 * h clockhour of halfday (1~12) number 12
	 *
	 * H hour of day (0~23) number 0
	 *
	 * k clockhour of day (1~24) number 24
	 *
	 * m minute of hour number 30
	 *
	 * s second of minute number 55
	 *
	 * S fraction of second number 978
	 *
	 * z time zone text Pacific Standard Time; PST
	 *
	 * Z time zone offset/id zone -0800; -08:00; America/Los_Angeles
	 *
	 * ' escape for text delimiter
	 *
	 * '' single quote literal '
	 */

	/**
	 * @param n
	 * @return added thousand-points IMPROVE use P13n to decide , or .
	 */
	public static String largeNumber(final long n) {
		final DecimalFormat formatter = (DecimalFormat) NumberFormat.getInstance(Locale.GERMANY);
		final DecimalFormatSymbols symbols = formatter.getDecimalFormatSymbols();
		symbols.setGroupingSeparator('.');
		formatter.setDecimalFormatSymbols(symbols);
		return formatter.format(n);
	}

	/**
	 * @param length
	 * @return 'xxx B' or 'xxx KB' or 'xxx MB' or 'xxx GB' or 'xxx TB', no
	 *         rounding
	 */
	public static String fileSize(final long length) {
		if (length == -1) {
			return "n/a";
		}
		if (length == 0) {
			return "Empty";
		}
		if (length < 1024) {
			return length + " B";
		}
		if (length < 1024 * 1024) {
			return length / 1024 + " KB";
		}
		final long length_in_mb = length / (1024 * 1024);
		if (length_in_mb < 1024) {
			return length_in_mb + " MB";
		}
		if (length_in_mb < 1024 * 1024) {
			return length_in_mb / 1024 + " GB";
		}

		return length_in_mb / (1024 * 1024) + " TB";
	}

	/**
	 * @param datetime
	 * @return year of datetime in UTC
	 */
	public static String year(final long datetime) {
		final DateTime dt = new DateTime(datetime, DateTimeZone.UTC);
		return dt.toString("yyyy");
	}

}
