package de.xam.textsearch.fragment;

import java.io.Serializable;
import java.util.Collections;
import java.util.List;
import java.util.Set;

import org.xydra.annotations.CanBeNull;
import org.xydra.index.iterator.ClosableIterator;
import org.xydra.index.iterator.IFilter;
import org.xydra.index.iterator.Iterators;
import org.xydra.index.query.Constraint;

import de.xam.textsearch.text.ITextQuery;

/**
 * A basic query to ask the {@link FragmentIndex}.
 *
 * Does no normalisation and no tokenization.
 *
 * @author xamde
 *
 * @param <V>
 */
public class FragmentQuery<V extends Serializable> implements Constraint<String>, ITextQuery<V> {

	/**
	 * @param fragmentIndex
	 * @param fragment
	 * @param prefixMatch default is false; true returns more results by
	 *            answering 'fragment*'
	 * @return a query with defaults: no prefix matches (aka exact search);
	 *         unlimited results;
	 */
	public static <X extends Serializable> FragmentQuery<X> create(final FragmentIndex<X> fragmentIndex, final String fragment,
			final boolean prefixMatch) {
		return new FragmentQuery<X>(fragmentIndex, fragment, prefixMatch, Integer.MAX_VALUE);
	}

	private final String fragment;

	private final FragmentIndex<V> fragmentIndex;

	private int maxResults;

	private IFilter<V> optionalValueFilter = null;

	/*
	 * iff true include prefix matches; otherwise return only exact, full
	 * matches. Default is false.
	 */
	private boolean prefixMatch;

	/**
	 * @param fragmentIndex
	 * @param fragment
	 * @param prefixMatch iff true include prefix matches; otherwise return only
	 *            exact, full matches
	 * @param maxResults use {@link Integer#MAX_VALUE} for all
	 */
	public FragmentQuery(final FragmentIndex<V> fragmentIndex, final String fragment, final boolean prefixMatch,
			final int maxResults) {
		super();
		this.fragmentIndex = fragmentIndex;
		this.fragment = fragment;
		this.prefixMatch = prefixMatch;
		this.maxResults = maxResults;
	}

	@Override
	public Set<V> executeToSet() {
		return this.fragmentIndex.executeQueryAsSet(this);
	}

	@Override
	public ClosableIterator<V> execute() {
		final ClosableIterator<V> it = this.fragmentIndex.executeQuery(this);
		return Iterators.distinct(it);
	}

	@Override
	public String getExpected() {
		return this.fragment;
	}

	public String getFragment() {
		return this.fragment;
	}

	public int getMaxResults() {
		return this.maxResults;
	}

	@CanBeNull
	public IFilter<V> getOptionalValueFilter() {
		return this.optionalValueFilter;
	}

	@Override
	public boolean isExact() {
		return !isPrefixMatch();
	}

	public boolean isPrefixMatch() {
		return this.prefixMatch;
	}

	@Override
	public boolean isStar() {
		return false;
	}

	@Override
	public boolean matches(final String element) {
		if (isPrefixMatch()) {
			return element.startsWith(this.fragment);
		} else {
			return element.equals(this.fragment);
		}
	}

	/**
	 * @param m default is unlimited
	 * @return this for a fluent API
	 */
	@Override
	public FragmentQuery<V> setMaxResults(final int m) {
		this.maxResults = m;
		return this;
	}

	/**
	 * @param b default is false
	 * @return this for a fluent API
	 */
	public FragmentQuery<V> setPrefixMatch(final boolean b) {
		this.prefixMatch = b;
		return this;
	}

	/**
	 * @param optionalValueFilter
	 * @CanBeNull
	 * @return this for a fluent API
	 */
	public FragmentQuery<V> setValueFilter(final IFilter<V> optionalValueFilter) {
		this.optionalValueFilter = optionalValueFilter;
		return this;
	}

	@Override
	public String getPhrase() {
		return this.fragment;
	}

	@Override
	public List<String> getWords() {
		return Collections.emptyList();
	}

	@Override
	public List<String> getFragments() {
		return Collections.singletonList(this.fragment);
	}

	@Override
	public String toString() {
		final StringBuilder b = new StringBuilder();
		b.append("'");
		b.append(this.fragment);
		b.append("'");
		b.append(isExact() ? "" : "*");
		if (isLimited()) {
			b.append(" ");
			b.append("[max=");
			b.append(getMaxResults());
			b.append("]");
		}
		if(this.optionalValueFilter!=null) {
			b.append(" filter:");
			b.append(this.optionalValueFilter.toString());
		}
		return b.toString();
	}

	public boolean isLimited() {
		return getMaxResults() < Integer.MAX_VALUE;
	}

	/**
	 * @return true if the optional value filter is set
	 */
	public boolean isFiltered() {
		return this.optionalValueFilter != null;
	}
}
