package de.xam.textsearch.util;

import java.util.Collection;
import java.util.HashSet;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.NoSuchElementException;
import java.util.Set;

import com.google.common.base.Function;
import com.google.gwt.regexp.shared.SplitResult;

import de.xam.textsearch.Match;

public class TextIndexTool {

	/**
	 * @param t
	 *            e.g. "Hello"
	 * @param minimalPrefixLength
	 *            e.g. 1
	 * @param maximalPrefixLength
	 *            e.g. 3
	 * @return substrings of given length, always starting at string index 0,
	 *         e.g. "H","He", an "Hel".
	 */
	public static List<String> generatePrefixes(final String t, final int minimalPrefixLength,
			final int maximalPrefixLength) {
		final List<String> result = new LinkedList<String>();
		if (t.length() < minimalPrefixLength) {
			return result;
		} else {
			final int maxPrefixLen = Math.min(t.length(), maximalPrefixLength);

			for (int i = minimalPrefixLength; i <= maxPrefixLen; i++) {
				final String subString = t.substring(0, i);
				result.add(subString);
			}
			return result;
		}
	}

	/**
	 * IMPROVE performance: make sure not to return the same strings as
	 * generated by prefix strings?
	 *
	 * @param token
	 * @param minimalSubStringLength
	 * @param maximalSubstringLength
	 * @return the set of all possible substrings in the given length range;
	 *         tokens which are too short to begin with are also returned as
	 *         resulting tokens
	 */
	public static Set<String> generateSubstrings(final String token, final int minimalSubStringLength,
			final int maximalSubstringLength) {
		final Set<String> result = new HashSet<String>();
		if (token.length() < minimalSubStringLength) {
			result.add(token);
			return result;
		} else {
			final int maxSubStringLen = Math.min(token.length(), maximalSubstringLength);

			for (int subStringLen = minimalSubStringLength; subStringLen <= maxSubStringLen; subStringLen++) {
				// how often can we walk t for a given subStringLen?
				final int slicesMax = token.length() - subStringLen + 1;
				for (int i = 0; i < slicesMax; i++) {
					final String subString = token.substring(i, i + subStringLen);
					result.add(subString);
				}
			}
			return result;
		}
	}

	public static class SplitResultIterator implements Iterator<String> {

		private final SplitResult data;

		private int cursor = 0;

		public SplitResultIterator(final SplitResult data) {
			this.data = data;
		}

		@Override
		public boolean hasNext() {
			return this.cursor < this.data.length();
		}

		@Override
		public String next() {
			if (hasNext()) {
				return this.data.get(this.cursor++);
			} else {
				throw new NoSuchElementException();
			}
		}

		@Override
		public void remove() {
			throw new UnsupportedOperationException();
		}

	}

	public static final Function<String, Void> DO_NOTHING = new Function<String, Void>() {

		@Override
		public Void apply(final String input) {
			return null;
		}
	};

	public static <V> Set<V> toSet(final Iterator<Match<V>> matches) {
		final Set<V> resSet = new HashSet<V>();
		while (matches.hasNext()) {
			final Match<V> match = matches.next();
			resSet.add(match.getValue());
		}
		return resSet;
	}

	/**
	 * @param searchResult
	 */
	public static <T> void dump(final Iterator<Match<T>> searchResult) {
		System.out.println("Match Iterator:");
		while (searchResult.hasNext()) {
			final Match<T> match = searchResult.next();
			System.out.println("Found val='" + match.getValue() + "' q=" + match.q() + " matched="
					+ match);
		}
	}

	public static int sumOfStringLengths(final Collection<String> stringSet) {
		int i = 0;
		for (final String s : stringSet) {
			i += s.length();
		}
		return i;
	}
}
