package de.xam.triplerules.impl;

import java.util.HashSet;
import java.util.Set;

import org.xydra.index.impl.DebugUtils;
import org.xydra.index.query.ITriple;

import de.xam.triplerules.IRuleConditionBinding;
import de.xam.triplerules.ITriplePattern;
import de.xam.triplerules.ITripleRule;
import de.xam.triplerules.IVariable;

/**
 * Manages variables and the values they are bound to. Manages also a set of
 * unbound patterns, in which the variables occur.
 *
 * @author xamde
 *
 * @param <K>
 * @param <L>
 * @param <M>
 */
public class ConditionBinding<K, L, M> implements IRuleConditionBinding<K, L, M> {

	private Object[] bindings;

	private Set<ITriplePattern<K, L, M>> unboundPatterns;

	@Override
	public <E> E boundValue(final IVariable<E> variable) {
		assert variable.isStar();
		return boundValue(variable.id());
	}

	/**
	 * Used in iteration 0 + n.
	 *
	 * Creates a binding, no checks are performed
	 *
	 * @param rule
	 * @param pattern
	 *            a part of the rules condition
	 * @param triple
	 *            input data
	 * @return a {@link ConditionBinding} matching the pattern and the triple.
	 *         The rule is used to manage efficient variable indices.
	 */
	public static <K, L, M> ConditionBinding<K, L, M> createInitialBinding(
			final ITripleRule<K, L, M> rule, final ITriplePattern<K, L, M> pattern, final ITriple<K, L, M> triple) {
		assert rule.condition().patterns().contains(pattern);
		final ConditionBinding<K, L, M> conditionBinding = new ConditionBinding<K, L, M>(rule, pattern);
		conditionBinding.bindInitially(pattern, triple);
		return conditionBinding;
	}

	private void bindInitially(final ITriplePattern<K, L, M> pattern, final ITriple<K, L, M> triple) {
		if (pattern.s().isStar()) {
			setValue(pattern.s(), triple.getKey1());
		} else {
			// no variable needs to be bound
		}

		if (pattern.p().isStar()) {
			setValue(pattern.p(), triple.getKey2());
		} else {
			// no variable needs to be bound
		}

		if (pattern.o().isStar()) {
			setValue(pattern.o(), triple.getEntry());
		} else {
			// no variable needs to be bound
		}
	}

	/**
	 * Create a condition binding with one initially bound pattern
	 *
	 * @param rule
	 * @param boundPattern
	 */
	public ConditionBinding(final ITripleRule<K, L, M> rule, final ITriplePattern<K, L, M> boundPattern) {
		this(rule.varCount());
		for (final ITriplePattern<K, L, M> pattern : rule.condition().patterns()) {
			if (!pattern.equals(boundPattern)) {
				this.unboundPatterns.add(pattern);
			}
		}
	}

	/**
	 * Create a condition binding with no bound variables (i.e. all unbound)
	 *
	 * @param capacity
	 *            total number of distinct variables in the rule
	 */
	private ConditionBinding(final int capacity) {
		this.bindings = new Object[capacity];
		this.unboundPatterns = new HashSet<ITriplePattern<K, L, M>>();
	}

	/**
	 * @param existing
	 * @param newBoundPattern
	 * @return a binding with one less unbound pattern
	 */
	public static <K, L, M> ConditionBinding<K, L, M> createRefined(
			final ConditionBinding<K, L, M> existing, final ITriplePattern<K, L, M> newBoundPattern) {
		final ConditionBinding<K, L, M> refined = new ConditionBinding<K, L, M>(existing.capacity());
		System.arraycopy(existing.bindings, 0, refined.bindings, 0, existing.capacity());

		for (final ITriplePattern<K, L, M> pattern : existing.unboundPatterns) {
			if (!pattern.equals(newBoundPattern)) {
				refined.unboundPatterns.add(pattern);
			}
		}

		return refined;
	}

	public <E> void setValue(final IVariable<E> variable, final Object value) {
		assert variable.isStar();
		this.bindings[variable.id()] = value;
	}

	public String toString(final ITripleRule<K, L, M> rule) {
		return toString(this, rule);
	}

	private static <K, L, M> String toString(final IRuleConditionBinding<K, L, M> binding,
			final ITripleRule<K, L, M> rule) {
		if (binding.size() == 0) {
			return "--no vars bound--";
		}

		final StringBuffer buf = new StringBuffer();
		buf.append("Bound=" + binding.size() + ": {");
		for (int i = 0; i < binding.capacity(); i++) {
			final Object o = binding.boundValue(i);
			final String oString = DebugUtils.toLimitedString(o, 20);
			if (rule == null) {
				final int name = i;
				buf.append("?" + name + "='" + oString + "'");
			} else {
				final String name = rule.translateIdToName(i);
				buf.append("?" + name + "-" + i + "='" + oString + "'");
			}
			if (i + 1 < binding.capacity()) {
				buf.append(", ");
			}
		}
		buf.append("}");
		return buf.toString();
	}

	@Override
	public String toString() {
		return toString(this, null);
	}

	@Override
	public int size() {
		int bound = 0;
		for (int i = 0; i < this.bindings.length; i++) {
			if (isBound(i)) {
				bound++;
			}
		}
		return bound;
	}

	@SuppressWarnings("unchecked")
	@Override
	public <E> E boundValue(final int i) {
		return (E) this.bindings[i];
	}

	@Override
	public int capacity() {
		return this.bindings.length;
	}

	@Override
	public <E> boolean isBound(final IVariable<E> variable) {
		return isBound(variable.id());
	}

	@Override
	public <E> boolean isBound(final int i) {
		return boundValue(i) != null;
	}

	public Set<ITriplePattern<K, L, M>> unboundPatterns() {
		return this.unboundPatterns;
	}

}
