package de.xam.triplerules.impl;

import java.io.Serializable;
import java.util.Collection;
import java.util.Iterator;

import org.xydra.index.ITripleIndex;
import org.xydra.index.query.ITriple;
import org.xydra.index.query.KeyKeyEntryTuple;
import org.xydra.log.api.Logger;
import org.xydra.log.api.LoggerFactory;

import de.xam.triplerules.IRuleConditionBinding;
import de.xam.triplerules.ITriplePattern;
import de.xam.triplerules.ITripleRule;
import de.xam.triplerules.IVariable;

/**
 * Use {@link #infLog} (logger on {@link de.xam.triplerules.impl.InfLog}) to see a reasoning trace
 *
 * @author xamde
 *
 */
public class RuleUtils {

	private static final Logger infLog = LoggerFactory.getLogger(InfLog.class);

	private static final Logger log = LoggerFactory.getLogger(RuleUtils.class);

	/**
	 * @param action
	 * @param pattern
	 * @return true iff the action matches the trigger pattern
	 */
	public static <K, L, M> boolean couldTrigger(final ITriplePattern<K, L, M> action,
			final ITriplePattern<K, L, M> pattern) {
		if (!couldTrigger(action.s(), pattern.s())) {
			return false;
		}
		if (!couldTrigger(action.p(), pattern.p())) {
			return false;
		}
		if (!couldTrigger(action.o(), pattern.o())) {
			return false;
		}

		return true;
	}

	/**
	 * @param ruleA
	 * @param ruleB
	 * @return true if any of the actions of ruleA matches any of the condition patterns of ruleB
	 */
	public static <K, L, M> boolean couldTrigger(final ITripleRule<K, L, M> ruleA, final ITripleRule<K, L, M> ruleB) {
		for (final ITriplePattern<K, L, M> action : ruleA.action().patterns()) {
			for (final ITriplePattern<K, L, M> pattern : ruleB.condition().patterns()) {
				if (couldTrigger(action, pattern)) {
					return true;
				}
			}
		}
		return false;
	}

	/**
	 * @param actionVar
	 * @param patternVar
	 * @return true iff the actionVar could be something that the patternVar matches on. This is only false, if the
	 *         patternVar is some fixed value and the actionVar is another fixed value
	 */
	public static <E> boolean couldTrigger(final IVariable<E> actionVar, final IVariable<E> patternVar) {
		final Object expectedPattern = patternVar.getExpected();
		// nothing expected? anything can match
		if (expectedPattern == null) {
			return true;
		}
		// anything can be generated? it could match.
		final Object generatedAction = actionVar.getExpected();
		if (generatedAction == null) {
			return true;
		}

		return expectedPattern.equals(generatedAction);
	}

	/**
	 * @param var
	 * @param binding
	 * @return
	 */
	@SuppressWarnings("unchecked")
	private static <K, L, M, E> E defineAction(final IVariable<E> var, final IRuleConditionBinding<K, L, M> binding) {
		Object o = var.getExpected();
		if (o == null) {
			o = binding.boundValue(var);
			assert o != null : "unbound variable: '" + var.name() + "'";
		}
		return (E) o;
	}

	/**
	 * Materialise into 'newInf'; adding to tripleIndex elsewhere.
	 *
	 * @param rule
	 * @param binding
	 * @param targetTripleIndex
	 * @param newInf
	 * @return number of inferred triples
	 */
	public static <K extends Serializable, L extends Serializable, M extends Serializable> int materialiseTriples(
			final ITripleRule<K, L, M> rule, final IRuleConditionBinding<K, L, M> binding,
			final ITripleIndex<K, L, M> targetTripleIndex, final Collection<ITriple<K, L, M>> newInf) {
		int inferredTriplesCount = 0;

		if (log.isDebugEnabled()) {
			log.debug(org.xydra.sharedutils.DebugUtils.toIndent("  ", binding.size() + 1)
					+ "Trying to materialise new triples from binding " + rule.condition().toString(binding));
		}

		for (final ITriplePattern<K, L, M> action : rule.action().patterns()) {
			final K s = defineAction(action.s(), binding);
			final L p = defineAction(action.p(), binding);
			final M o = defineAction(action.o(), binding);
			// avoid infinite recursion
			if (!targetTripleIndex.contains(s, p, o)) {
				final KeyKeyEntryTuple<K, L, M> newTriple = new KeyKeyEntryTuple<K, L, M>(s, p, o);
				newInf.add(newTriple);

				if (infLog.isDebugEnabled()) {

					final StringBuilder b = new StringBuilder();
					final Iterator<ITriplePattern<K, L, M>> pIt = rule.condition().patterns().iterator();
					boolean first = true;
					while (pIt.hasNext()) {
						final ITriplePattern<K, L, M> tp = pIt.next();
						if (first) {
							b.append("   ");
						} else {
							b.append(" + ");
						}
						first = false;
						b.append(tp.toString(binding));
						b.append("\n");
					}
					b.append(" ----- " + rule.label() + " ----- => \n");

					final Iterator<ITriplePattern<K, L, M>> aIt = rule.action().patterns().iterator();
					while (aIt.hasNext()) {
						final ITriplePattern<K, L, M> tp = aIt.next();
						b.append("   ");
						b.append(tp.toString(binding));
						b.append("\n");
					}

					b.append("Inferred Triple: "+newTriple);
					infLog.debug("\n" + b.toString());
				}
				if (log.isDebugEnabled()) {
					log.debug(org.xydra.sharedutils.DebugUtils.toIndent("++", binding.size() + 1) + "Inferred "
							+ newTriple);
				}
				inferredTriplesCount++;
			}
		}
		return inferredTriplesCount;
	}
}
