package org.xydra.core.model.impl.memory;

import java.io.Serializable;

import org.xydra.annotations.CanBeNull;
import org.xydra.annotations.ModificationOperation;
import org.xydra.annotations.ReadOperation;
import org.xydra.base.Base;
import org.xydra.base.XAddress;
import org.xydra.base.XId;
import org.xydra.base.XType;
import org.xydra.base.change.XCommand;
import org.xydra.base.change.XFieldCommand;
import org.xydra.base.change.impl.memory.MemoryFieldCommand;
import org.xydra.base.rmof.XRevWritableField;
import org.xydra.base.rmof.XRevWritableModel;
import org.xydra.base.rmof.XRevWritableObject;
import org.xydra.base.rmof.impl.XExistsRevWritableField;
import org.xydra.base.rmof.impl.memory.SimpleField;
import org.xydra.base.value.XValue;
import org.xydra.core.XCopyUtils;
import org.xydra.core.change.XFieldEventListener;
import org.xydra.core.model.XField;
import org.xydra.core.model.XObject;
import org.xydra.core.model.impl.memory.sync.Root;
import org.xydra.sharedutils.XyAssert;


/**
 * An implementation of {@link XField}.
 *
 * @author xamde
 * @author kaidel
 */
public class MemoryField extends AbstractMOFEntity implements XField, IMemoryField, Serializable,
        Synchronizable {

    private static final long serialVersionUID = -4390811955475742528L;

    /**
     * the father-object which is holding this field (may be null). Used
     * properly execute field commands.
     */
    @CanBeNull
    private final IMemoryObject father;

    /** The internal runtime state, like a snapshot */
    private final XExistsRevWritableField fieldState;

    /**
     * Creates a new MemoryField with a father-{@link XObject}.
     *
     * @param father The father-{@link XObject} of this MemoryField @NeverNull
     * @param fieldState A {@link XRevWritableField} representing the initial
     *            state of this field. The {@link XField} will continue using
     *            this state object, so it must not be modified directly after
     *            wrapping it in an {@link XField} @NeverNull
     */
    public MemoryField(final IMemoryObject father, final XRevWritableField fieldState) {
        super(father.getRoot());
        if(fieldState == null) {
            throw new IllegalArgumentException("fieldState may not be null");
        }
        assert fieldState.getAddress().getRepository() != null;
        assert fieldState.getAddress().getModel() != null;
        assert fieldState.getAddress().getObject() != null;

        this.fieldState = convert(fieldState);
        this.father = father;
    }

    private static XExistsRevWritableField convert(final XRevWritableField fieldState) {
        if(fieldState instanceof XExistsRevWritableField) {
            return (XExistsRevWritableField)fieldState;
        } else {
            return XCopyUtils.createSnapshot(fieldState);
        }
    }

    /**
     * Create a stand-alone field that exists
     *
     * This instance cannot be added directly to another object. However, the
     * internal state can be added to another object.
     *
     * @param actorId The actor to be used in events generated by this field.
     * @param fieldId The {@link XId} for this MemoryField.
     */
    public MemoryField(final XId actorId, final XId fieldId) {
        this(actorId, new SimpleField(Base.toAddress(XId.DEFAULT, XId.DEFAULT, XId.DEFAULT, fieldId)));
    }

    /**
     * Create a stand-alone field that exists
     *
     * This instance cannot be added directly to another object. However, the
     * internal state can be added to another object.
     *
     * @param actorId The actor to be used in events generated by this field.
     * @param fieldState A {@link XRevWritableField} representing the initial
     *            state of this field. The {@link XField} will continue using
     *            this state object, so it must not be modified directly after
     *            wrapping it in an {@link XField} @NeverNull
     */
    public MemoryField(final XId actorId, final XRevWritableField fieldState) {
        super(Root.createWithActor(actorId, fieldState.getAddress(), XCommand.NONEXISTANT));
        assert fieldState.getAddress().getRepository() != null;
        assert fieldState.getAddress().getModel() != null;
        assert fieldState.getAddress().getObject() != null;
        this.fieldState = convert(fieldState);
        this.fieldState.setExists(true);
        this.father = null;
    }

    /**
     * Adds the given {@link XFieldEventListener} to this MemoryField, if
     * possible.
     *
     * @param changeListener The {@link XFieldEventListener} which is to be
     *            added
     * @return false, if the given {@link XFieldEventListener} was already
     *         registered on this MemoryField, true otherwise
     */
    @Override
    public boolean addListenerForFieldEvents(final XFieldEventListener changeListener) {
        return getRoot().addListenerForFieldEvents(getAddress(), changeListener);
    }

    @Override
    synchronized public XRevWritableField createSnapshot() {
        if(!exists()) {
            return null;
        }
        return XCopyUtils.createSnapshot(this);
    }

    @Override
    @ReadOperation
    public boolean equals(final Object object) {
        synchronized(getRoot()) {
            return super.equals(object);
        }
    }

    @Override
    public long executeFieldCommand(final XFieldCommand command) {
        synchronized(getRoot()) {
            assertThisEntityExists();

            XRevWritableObject objectState = null;
            XRevWritableModel modelState = null;
            if(this.father != null) {
                objectState = this.father.getState();
                if(this.father.getFather() != null) {
                    modelState = this.father.getFather().getState();
                }
            }

            assert modelState == null || objectState == null
                    || modelState.getRevisionNumber() >= objectState.getRevisionNumber() :

            "modelRev smaller than objectRev??? " +

            "modelRev=" + modelState.getRevisionNumber()

            + " objectRev=" + objectState.getRevisionNumber();

            return Executor.executeCommandOnField(getRoot().getSessionActor(), command, modelState,
                    objectState, this.fieldState, getRoot(), null);
        }
    }

    @Override
    public XAddress getAddress() {
        synchronized(getRoot()) {
            return this.fieldState.getAddress();
        }
    }

    /**
     * @throws IllegalStateException if this method is called after this
     *             MemoryField was already removed
     */
    @Override
    @ReadOperation
    public XId getId() {
        synchronized(getRoot()) {
            return this.fieldState.getId();
        }
    }

    /**
     * @throws IllegalStateException if this method is called after this
     *             MemoryField was already removed
     */
    @Override
    @ReadOperation
    public long getRevisionNumber() {
        synchronized(getRoot()) {
            return this.fieldState.getRevisionNumber();
        }
    }

    @Override
    public Root getRoot() {
        return super.getRoot();
    }

    @Override
    public XId getSessionActor() {
        synchronized(getRoot()) {
            return getRoot().getSessionActor();
        }
    }

    @Override
    public XType getType() {
        return XType.XFIELD;
    }

    /**
     * @throws IllegalStateException if this method is called after this
     *             MemoryField was already removed
     */
    @Override
    @ReadOperation
    public XValue getValue() {
        synchronized(getRoot()) {
            assertThisEntityExists();
            return this.fieldState.getValue();
        }
    }

    @Override
    public int hashCode() {
        synchronized(getRoot()) {
            return super.hashCode();
        }
    }

    @Override
    public boolean isEmpty() {
        synchronized(getRoot()) {
            assertThisEntityExists();
            return getValue() == null;
        }
    }

    @Override
    public boolean isSynchronized() {
        assert this.father != null : "A standalone MemoryField can't be synchronized.";

        /*
         * If the field's revNo is smaller than its father's syncRevNo, the
         * field is synchronized.
         */
        return getRevisionNumber() <= getRoot().getSynchronizedRevision();
    }

    /**
     * Removes the given {@link XFieldEventListener} from this MemoryField.
     *
     * @param changeListener The {@link XFieldEventListener} which is to be
     *            removed
     * @return true, if the given {@link XFieldEventListener} was registered on
     *         this MemoryField, false otherwise
     */

    @Override
    public boolean removeListenerForFieldEvents(final XFieldEventListener changeListener) {
        return getRoot().removeListenerForFieldEvents(getAddress(), changeListener);
    }

    @Override
    public void setSessionActor(final XId actorId) {
        getRoot().setSessionActor(actorId);
    }

    /**
     * @throws IllegalStateException if this method is called after this
     *             MemoryField was already removed
     */
    @Override
    @ModificationOperation
    public boolean setValue(final XValue newValue) {
        /*
         * no synchronization necessary here (except that in
         * executeFieldCommand())
         */
        XFieldCommand command;
        if(newValue == null) {
            command = MemoryFieldCommand.createRemoveCommand(getAddress(), XCommand.FORCED);
        } else {
            command = MemoryFieldCommand.createAddCommand(getAddress(), XCommand.FORCED, newValue);
        }

        final long result = executeFieldCommand(command);
        XyAssert.xyAssert(result >= 0 || result == XCommand.NOCHANGE);
        return result != XCommand.NOCHANGE;
    }

    @Override
    public String toString() {
        return getId() + " rev[" + getRevisionNumber() + "]";
    }

    @Override
    public boolean exists() {
        return this.fieldState.exists();
    }

    @Override
    public void setExists(final boolean entityExists) {
        this.fieldState.setExists(entityExists);
    }

    @Override
    public long executeCommand(final XCommand command) {
        if(command instanceof XFieldCommand) {
            return executeFieldCommand((XFieldCommand)command);
        } else {
            throw new IllegalArgumentException("Can only execute field commands on fields");
        }
    }

}
